<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class CustomSession extends Model
{
    use HasFactory;

    protected $fillable = [
        'custom_request_id',
        'user_id',
        'title',
        'description',
        'file_path',
        'transcript',
        'duration',
        'status',
        'generation_metadata',
        'generated_at',
    ];

    protected $casts = [
        'generation_metadata' => 'array',
        'generated_at' => 'datetime',
    ];

    public function customRequest(): BelongsTo
    {
        return $this->belongsTo(CustomRequest::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    public function isQueued(): bool
    {
        return $this->status === 'queued';
    }

    public function isGenerating(): bool
    {
        return $this->status === 'generating';
    }

    public function isReady(): bool
    {
        return $this->status === 'ready';
    }

    public function isFailed(): bool
    {
        return $this->status === 'failed';
    }

    /**
     * Get the signed URL for the audio file
     */
    public function getAudioUrlAttribute(): ?string
    {
        if (empty($this->file_path)) {
            return null;
        }

        $mediaService = app(\App\Services\MediaService::class);
        return $mediaService->generateSignedUrl($this->file_path, 60); // 1 hour TTL
    }

    /**
     * Get the public URL if using public disk
     */
    public function getPublicAudioUrlAttribute(): ?string
    {
        if (empty($this->file_path)) {
            return null;
        }

        $mediaService = app(\App\Services\MediaService::class);
        return $mediaService->getPublicUrl($this->file_path);
    }
}
