<?php

use Illuminate\Support\Facades\Route;
use App\Models\Season;
use App\Models\Category;
use App\Models\Item;

// Admin routes
Route::middleware(['auth', 'verified', 'admin'])->prefix('admin')->name('admin.')->group(function () {
    Route::get('/', function () {
        return view('admin.dashboard');
    })->name('dashboard');

    // Seasons CRUD (basic placeholder routes)
    Route::get('/seasons', [\App\Http\Controllers\Admin\SeasonController::class, 'index'])->name('seasons.index');
    Route::get('/seasons/create', [\App\Http\Controllers\Admin\SeasonController::class, 'create'])->name('seasons.create');
    Route::post('/seasons', [\App\Http\Controllers\Admin\SeasonController::class, 'store'])->name('seasons.store');
    Route::get('/seasons/{season}/edit', [\App\Http\Controllers\Admin\SeasonController::class, 'edit'])->name('seasons.edit');
    Route::put('/seasons/{season}', [\App\Http\Controllers\Admin\SeasonController::class, 'update'])->name('seasons.update');
    Route::delete('/seasons/{season}', [\App\Http\Controllers\Admin\SeasonController::class, 'destroy'])->name('seasons.destroy');

    // Categories & Items
    Route::get('/categories', [\App\Http\Controllers\Admin\CategoryController::class, 'index'])->name('categories.index');
    Route::get('/items', [\App\Http\Controllers\Admin\ItemController::class, 'index'])->name('items.index');
    Route::get('/items/create', [\App\Http\Controllers\Admin\ItemController::class, 'create'])->name('items.create');
    Route::post('/items', [\App\Http\Controllers\Admin\ItemController::class, 'store'])->name('items.store');
    Route::get('/items/{item}/edit', [\App\Http\Controllers\Admin\ItemController::class, 'edit'])->name('items.edit');
    Route::put('/items/{item}', [\App\Http\Controllers\Admin\ItemController::class, 'update'])->name('items.update');
});
use App\Models\Customer;
use App\Models\Order;

Route::view('/', 'welcome');

Route::view('dashboard', 'dashboard')
    ->middleware(['auth', 'verified'])
    ->name('dashboard');

Route::view('profile', 'profile')
    ->middleware(['auth'])
    ->name('profile');

// Test routes for the new models
Route::get('/test/seasons', function () {
    $seasons = Season::with('items')->get();
    return response()->json($seasons);
});

Route::get('/test/categories', function () {
    $categories = Category::with('items')->ordered()->get();
    return response()->json($categories);
});

Route::get('/test/items', function () {
    $items = Item::with(['category', 'season', 'colourways'])
        ->active()
        ->get();
    return response()->json($items);
});

Route::get('/test/customers', function () {
    $customers = Customer::with(['addresses', 'orders'])->get();
    return response()->json($customers);
});

Route::get('/test/orders', function () {
    $orders = Order::with(['customer', 'orderLines.item', 'orderLines.colourway'])
        ->get();
    return response()->json($orders);
});

// Collections page
Route::get('/collections', function () {
    $activeSeasons = Season::where('is_active', true)
        ->orderBy('year')
        ->orderBy('half', 'desc')
        ->get();
    
    return view('collections', compact('activeSeasons'));
})->name('collections');

// Individual collection page (dynamic)
Route::get('/collections/{season}', function (string $seasonCode) {
    $season = Season::where('code', $seasonCode)
        ->where('is_active', true)
        ->firstOrFail();
    
    $items = Item::with(['category', 'season', 'colourways'])
        ->where('season_id', $season->id)
        ->where('is_active', true)
        ->orderBy('name')
        ->get();
    
    $categories = Category::whereHas('items', function($query) use ($season) {
        $query->where('season_id', $season->id);
    })->withCount(['items' => function($query) use ($season) {
        $query->where('season_id', $season->id);
    }])->get();
    
    $relatedSeasons = Season::where('is_active', true)
        ->where('id', '!=', $season->id)
        ->orderBy('year')
        ->orderBy('half', 'desc')
        ->take(3)
        ->get();
    
    return view('collection-detail', compact('season', 'items', 'categories', 'relatedSeasons'));
})->name('collection.detail');

require __DIR__.'/auth.php';
