<?php

use Illuminate\Support\Facades\Route;
use App\Models\Season;
use App\Models\Category;
use App\Models\Item;
use App\Models\Customer;
use App\Models\Order;

Route::view('/', 'welcome');

Route::view('dashboard', 'dashboard')
    ->middleware(['auth', 'verified'])
    ->name('dashboard');

Route::view('profile', 'profile')
    ->middleware(['auth'])
    ->name('profile');

// Test routes for the new models
Route::get('/test/seasons', function () {
    $seasons = Season::with('items')->get();
    return response()->json($seasons);
});

Route::get('/test/categories', function () {
    $categories = Category::with('items')->ordered()->get();
    return response()->json($categories);
});

Route::get('/test/items', function () {
    $items = Item::with(['category', 'season', 'colourways'])
        ->active()
        ->get();
    return response()->json($items);
});

Route::get('/test/customers', function () {
    $customers = Customer::with(['addresses', 'orders'])->get();
    return response()->json($customers);
});

Route::get('/test/orders', function () {
    $orders = Order::with(['customer', 'orderLines.item', 'orderLines.colourway'])
        ->get();
    return response()->json($orders);
});

// Collections page
Route::get('/collections', function () {
    $activeSeasons = Season::where('is_active', true)
        ->orderBy('year')
        ->orderBy('half', 'desc')
        ->get();
    
    $featuredCategories = Category::where('sort_order', '>', 0)
        ->orderBy('sort_order')
        ->take(6)
        ->get();
    
    $latestItems = Item::with(['category', 'season', 'colourways'])
        ->where('is_active', true)
        ->latest()
        ->take(12)
        ->get();
    
    return view('collections', compact('activeSeasons', 'featuredCategories', 'latestItems'));
})->name('collections');

// Individual collection page (dynamic)
Route::get('/collections/{season}', function (string $seasonCode) {
    $season = Season::where('code', $seasonCode)
        ->where('is_active', true)
        ->firstOrFail();
    
    $items = Item::with(['category', 'season', 'colourways'])
        ->where('season_id', $season->id)
        ->where('is_active', true)
        ->orderBy('name')
        ->get();
    
    $categories = Category::whereHas('items', function($query) use ($season) {
        $query->where('season_id', $season->id);
    })->withCount(['items' => function($query) use ($season) {
        $query->where('season_id', $season->id);
    }])->get();
    
    $relatedSeasons = Season::where('is_active', true)
        ->where('id', '!=', $season->id)
        ->orderBy('year')
        ->orderBy('half', 'desc')
        ->take(3)
        ->get();
    
    return view('collection-detail', compact('season', 'items', 'categories', 'relatedSeasons'));
})->name('collection.detail');

require __DIR__.'/auth.php';
