<?php

namespace App\Services\VAT;

use App\Models\ReceiptLine;
use App\Models\VatAnalysis;
use App\Services\AI\VertexClient;

class UkVatService
{
    public function __construct(private VertexClient $vertex) {}

    public function analyzeLine(ReceiptLine $line): VatAnalysis
    {
        $vatRate = $line->vat_rate ?? 20.0;
        $claimable = 0.0;

        if ($line->is_personal) {
            $claimable = 0.0;
        } else {
            // Ask Vertex for UK VAT claim guidance
            $model = config('vertex.models.vat', 'gemini-1.5-flash');
            $system = 'You are a UK VAT expert. For the given receipt line, determine the VAT rate and the claimable VAT amount under UK VAT law (ignore personal items). Output JSON with vat_rate and vat_amount_claimable and a short reasoning.';
            $prompt = json_encode([
                'description' => $line->description,
                'line_total' => $line->line_total,
                'vat_rate_hint' => $line->vat_rate,
                'is_personal' => $line->is_personal,
                'department' => $line->department?->name,
                'account' => $line->account?->name,
            ]);
            $resp = $this->vertex->generate($model, $prompt, $system, ['responseMimeType' => 'application/json']);
            $json = $resp['json'];
            if (is_array($json)) {
                if (isset($json['vat_rate'])) $vatRate = (float) $json['vat_rate'];
                if (isset($json['vat_amount_claimable'])) $claimable = (float) $json['vat_amount_claimable'];
                $reason = $json['reasoning'] ?? 'LLM suggested.';
            }

            // Fallback baseline
            if (!$claimable) {
                if ($line->vat_amount !== null) {
                    $claimable = (float) $line->vat_amount;
                } else {
                    $claimable = round(($line->line_total * ($vatRate / (100 + $vatRate))), 2);
                }
            }
        }

        return VatAnalysis::updateOrCreate(
            ['receipt_line_id' => $line->id],
            [
                'jurisdiction' => 'UK',
                'vat_rate' => $vatRate,
                'vat_amount_claimable' => $claimable,
                'reasoning' => $reason ?? 'Baseline UK VAT calculation.',
            ]
        );
    }
}


