<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Session;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class SessionController extends Controller
{
    /**
     * Display a listing of sessions
     */
    public function index(Request $request)
    {
        $sessions = $request->user()->sessions()
            ->with(['meditation'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return response()->json([
            'success' => true,
            'data' => $sessions
        ]);
    }

    /**
     * Store a newly created session
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'duration' => 'required|integer|min:1|max:180',
            'type' => 'required|string|in:relaxation,anxiety,stress,sleep,confidence,weight_loss,smoking_cessation',
            'difficulty' => 'required|string|in:beginner,intermediate,advanced',
            'meditation_id' => 'nullable|exists:meditations,id',
            'custom_script' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $session = $request->user()->sessions()->create([
            'title' => $request->title,
            'description' => $request->description,
            'duration' => $request->duration,
            'type' => $request->type,
            'difficulty' => $request->difficulty,
            'meditation_id' => $request->meditation_id,
            'custom_script' => $request->custom_script,
            'status' => 'created',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Session created successfully',
            'data' => $session->load('meditation')
        ], 201);
    }

    /**
     * Display the specified session
     */
    public function show(Session $session)
    {
        if ($session->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        return response()->json([
            'success' => true,
            'data' => $session->load('meditation')
        ]);
    }

    /**
     * Update the specified session
     */
    public function update(Request $request, Session $session)
    {
        if ($session->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'title' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'duration' => 'sometimes|required|integer|min:1|max:180',
            'type' => 'sometimes|required|string|in:relaxation,anxiety,stress,sleep,confidence,weight_loss,smoking_cessation',
            'difficulty' => 'sometimes|required|string|in:beginner,intermediate,advanced',
            'meditation_id' => 'nullable|exists:meditations,id',
            'custom_script' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $session->update($request->only([
            'title', 'description', 'duration', 'type', 
            'difficulty', 'meditation_id', 'custom_script'
        ]));

        return response()->json([
            'success' => true,
            'message' => 'Session updated successfully',
            'data' => $session->load('meditation')
        ]);
    }

    /**
     * Remove the specified session
     */
    public function destroy(Session $session)
    {
        if ($session->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $session->delete();

        return response()->json([
            'success' => true,
            'message' => 'Session deleted successfully'
        ]);
    }

    /**
     * Start a session
     */
    public function startSession(Session $session)
    {
        if ($session->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $session->update([
            'status' => 'in_progress',
            'started_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Session started successfully',
            'data' => $session
        ]);
    }

    /**
     * Complete a session
     */
    public function completeSession(Request $request, Session $session)
    {
        if ($session->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'completion_percentage' => 'required|integer|min:0|max:100',
            'mood_before' => 'nullable|integer|min:1|max:10',
            'mood_after' => 'nullable|integer|min:1|max:10',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $session->update([
            'status' => 'completed',
            'completed_at' => now(),
            'completion_percentage' => $request->completion_percentage,
            'mood_before' => $request->mood_before,
            'mood_after' => $request->mood_after,
            'notes' => $request->notes,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Session completed successfully',
            'data' => $session
        ]);
    }

    /**
     * Get session progress
     */
    public function getProgress(Session $session)
    {
        if ($session->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $progress = [
            'session' => $session,
            'duration_completed' => $session->started_at ? now()->diffInSeconds($session->started_at) : 0,
            'total_duration' => $session->duration * 60, // Convert to seconds
            'progress_percentage' => $session->started_at ? 
                min(100, (now()->diffInSeconds($session->started_at) / ($session->duration * 60)) * 100) : 0,
        ];

        return response()->json([
            'success' => true,
            'data' => $progress
        ]);
    }
}
