<?php

namespace App\Services\AI;

use App\Models\StatementTransaction;
use App\Models\Receipt;
use App\Models\ReceiptLine;
use App\Models\ClassificationRule;
use App\Models\Department;
use App\Models\Account;
use App\Services\AI\VertexClient;
use Illuminate\Support\Facades\Log;

class ExpenseClassificationAgent
{
    public function __construct(private VertexClient $vertex) {}

    /**
     * Classify a statement transaction with intelligent department and account assignment
     */
    public function classifyTransaction(StatementTransaction $transaction, ?Receipt $receipt = null): array
    {
        $model = config('vertex.models.classify', 'gemini-2.5-flash-lite');
        
        // Build context from transaction and optional receipt
        $context = $this->buildClassificationContext($transaction, $receipt);
        
        // Get user's classification history for personalization
        $userHistory = $this->getUserClassificationHistory($transaction->user_id);
        
        $system = $this->buildSystemPrompt($userHistory);
        $prompt = $this->buildClassificationPrompt($context, $userHistory);

        $response = $this->vertex->generate($model, $prompt, $system, ['responseMimeType' => 'application/json']);
        
        if (!isset($response['json']) || !is_array($response['json'])) {
            Log::warning('ExpenseClassificationAgent: Invalid LLM response', [
                'transaction_id' => $transaction->id,
                'response' => $response
            ]);
            return $this->getFallbackClassification($transaction);
        }

        $classification = $response['json'];
        
        // Validate and apply classification
        $result = $this->applyClassification($transaction, $classification, $receipt);
        
        // Learn from this classification for future personalization
        $this->learnFromClassification($transaction, $classification, $receipt);
        
        return $result;
    }

    /**
     * Re-classify transaction when receipt is matched
     */
    public function reclassifyWithReceipt(StatementTransaction $transaction, Receipt $receipt): array
    {
        return $this->classifyTransaction($transaction, $receipt);
    }

    /**
     * Check if transaction should be split based on receipt analysis
     */
    public function shouldSplitTransaction(StatementTransaction $transaction, Receipt $receipt): array
    {
        $model = config('vertex.models.classify', 'gemini-2.5-flash-lite');
        
        $system = 'You are an expert at analyzing business expenses. Determine if a statement transaction should be split into multiple lines based on the receipt details. Consider:
        - Different departments (e.g., fuel vs meals)
        - Different account codes (e.g., travel vs entertainment)
        - Mixed personal/business expenses
        - Different VAT treatments
        
        Output JSON with:
        - should_split: boolean
        - split_lines: array of line objects with amount, department, account, description, is_personal
        - reasoning: explanation of the split decision';

        $prompt = json_encode([
            'transaction' => [
                'merchant_name' => $transaction->merchant_name,
                'amount' => $transaction->amount,
                'date' => $transaction->transaction_date,
                'mcc_description' => $transaction->mcc_description,
                'merchant_category' => $transaction->merchant_category,
            ],
            'receipt' => [
                'merchant_name' => $receipt->merchant_name,
                'total_amount' => $receipt->total_amount,
                'lines' => $receipt->lines()->get(['description', 'line_total', 'vat_rate'])->toArray(),
                'meta' => $receipt->meta,
            ],
            'allowed_departments' => $this->getAllowedDepartments(),
            'allowed_accounts' => $this->getAllowedAccounts(),
        ]);

        $response = $this->vertex->generate($model, $prompt, $system, ['responseMimeType' => 'application/json']);
        
        if (!isset($response['json']) || !is_array($response['json'])) {
            return ['should_split' => false, 'split_lines' => [], 'reasoning' => 'Unable to analyze receipt'];
        }

        return $response['json'];
    }

    /**
     * Build comprehensive context for classification
     */
    private function buildClassificationContext(StatementTransaction $transaction, ?Receipt $receipt = null): array
    {
        $context = [
            'transaction' => [
                'merchant_name' => $transaction->merchant_name,
                'amount' => $transaction->amount,
                'currency' => $transaction->currency,
                'date' => $transaction->transaction_date,
                'time' => $transaction->transaction_time,
                'mcc' => $transaction->mcc,
                'mcc_description' => $transaction->mcc_description,
                'merchant_category' => $transaction->merchant_category,
                'merchant_city' => $transaction->merchant_city,
                'merchant_state' => $transaction->merchant_state,
                'merchant_postcode' => $transaction->merchant_postcode,
                'transaction_type' => $transaction->transaction_type,
                'authorisation_code' => $transaction->authorisation_code,
                'meta' => $transaction->meta,
            ],
            'allowed_departments' => $this->getAllowedDepartments(),
            'allowed_accounts' => $this->getAllowedAccounts(),
        ];

        if ($receipt) {
            $context['receipt'] = [
                'merchant_name' => $receipt->merchant_name,
                'total_amount' => $receipt->total_amount,
                'currency' => $receipt->currency,
                'date' => $receipt->receipt_date,
                'lines' => $receipt->lines()->get(['description', 'line_total', 'quantity', 'unit_price', 'vat_rate'])->toArray(),
                'vat_number' => $receipt->meta['vat_number'] ?? null,
                'vat_amount' => $receipt->meta['vat_amount'] ?? null,
                'subtotal' => $receipt->meta['subtotal'] ?? null,
                'meta' => $receipt->meta,
            ];
        }

        return $context;
    }

    /**
     * Get user's classification history for personalization
     */
    private function getUserClassificationHistory(?int $userId): array
    {
        if (!$userId) {
            return [];
        }

        // Get recent classification patterns from this user
        $recentTransactions = StatementTransaction::where('user_id', $userId)
            ->whereNotNull('department_id')
            ->whereNotNull('account_id')
            ->orderBy('updated_at', 'desc')
            ->limit(50)
            ->get(['merchant_name', 'amount', 'department_id', 'account_id', 'is_personal', 'mcc_description']);

        $patterns = [];
        foreach ($recentTransactions as $txn) {
            $key = strtolower(trim($txn->merchant_name));
            if (!isset($patterns[$key])) {
                $patterns[$key] = [];
            }
            $patterns[$key][] = [
                'department_id' => $txn->department_id,
                'account_id' => $txn->account_id,
                'is_personal' => $txn->is_personal,
                'amount_range' => $this->getAmountRange($txn->amount),
            ];
        }

        // Get user-specific classification rules
        $userRules = ClassificationRule::where('user_id', $userId)
            ->orderBy('weight', 'desc')
            ->get(['pattern', 'department_id', 'account_id', 'is_personal', 'weight']);

        return [
            'patterns' => $patterns,
            'rules' => $userRules->toArray(),
        ];
    }

    /**
     * Build system prompt with user personalization
     */
    private function buildSystemPrompt(array $userHistory): string
    {
        $basePrompt = 'You are an expert at classifying business expenses for UK accounting. Your task is to assign the most appropriate department and account code to each transaction.

CLASSIFICATION GUIDELINES:
- Consider the merchant name, amount, date, location, and any receipt details
- Use UK business expense categories and VAT considerations
- Be consistent with similar transactions
- Consider the business context and typical expense patterns
- Mark as personal if clearly not business-related

OUTPUT FORMAT:
Return JSON with:
- department: string (must match allowed departments exactly)
- account: string (must match allowed accounts exactly)  
- is_personal: boolean
- confidence: float (0-1)
- reasoning: string explaining the classification
- suggested_split: boolean (if receipt suggests multiple expense types)
- split_details: array of split suggestions if applicable';

        if (!empty($userHistory['patterns'])) {
            $basePrompt .= "\n\nUSER PREFERENCE PATTERNS:\n";
            $basePrompt .= "This user has previously classified similar merchants as follows:\n";
            foreach ($userHistory['patterns'] as $merchant => $classifications) {
                $basePrompt .= "- {$merchant}: " . json_encode($classifications) . "\n";
            }
        }

        if (!empty($userHistory['rules'])) {
            $basePrompt .= "\n\nUSER-SPECIFIC RULES:\n";
            foreach ($userHistory['rules'] as $rule) {
                $basePrompt .= "- Pattern: {$rule['pattern']} -> Department: {$rule['department_id']}, Account: {$rule['account_id']}, Personal: " . ($rule['is_personal'] ? 'Yes' : 'No') . "\n";
            }
        }

        return $basePrompt;
    }

    /**
     * Build classification prompt
     */
    private function buildClassificationPrompt(array $context, array $userHistory): string
    {
        return json_encode([
            'transaction_data' => $context['transaction'],
            'receipt_data' => $context['receipt'] ?? null,
            'allowed_departments' => $context['allowed_departments'],
            'allowed_accounts' => $context['allowed_accounts'],
            'classification_context' => [
                'has_receipt' => !empty($context['receipt']),
                'merchant_confidence' => $this->calculateMerchantConfidence($context['transaction']['merchant_name']),
                'amount_significance' => $this->getAmountSignificance($context['transaction']['amount']),
            ],
        ]);
    }

    /**
     * Apply classification to transaction
     */
    private function applyClassification(StatementTransaction $transaction, array $classification, ?Receipt $receipt = null): array
    {
        $updates = [];
        $result = [
            'success' => false,
            'updates' => [],
            'confidence' => $classification['confidence'] ?? 0.5,
            'reasoning' => $classification['reasoning'] ?? 'No reasoning provided',
        ];

        // Map department name to ID
        if (!empty($classification['department'])) {
            $departmentId = $this->mapDepartmentNameToId($classification['department']);
            if ($departmentId) {
                $updates['department_id'] = $departmentId;
                $result['updates']['department'] = $classification['department'];
            }
        }

        // Map account name to ID
        if (!empty($classification['account'])) {
            $accountId = $this->mapAccountNameToId($classification['account']);
            if ($accountId) {
                $updates['account_id'] = $accountId;
                $result['updates']['account'] = $classification['account'];
            }
        }

        // Set personal flag
        if (isset($classification['is_personal'])) {
            $updates['is_personal'] = (bool) $classification['is_personal'];
            $result['updates']['is_personal'] = $classification['is_personal'];
        }

        // Apply updates if any
        if (!empty($updates)) {
            $transaction->update($updates);
            $result['success'] = true;
        }

        // Handle suggested splits
        if (!empty($classification['suggested_split']) && $classification['suggested_split'] && $receipt) {
            $result['split_suggestion'] = $classification['split_details'] ?? [];
        }

        return $result;
    }

    /**
     * Learn from classification for future personalization
     */
    private function learnFromClassification(StatementTransaction $transaction, array $classification, ?Receipt $receipt = null): void
    {
        if (!$transaction->user_id) {
            return;
        }

        // Create or update classification rule based on this decision
        $pattern = $this->generatePatternFromTransaction($transaction);
        
        if ($pattern) {
            ClassificationRule::updateOrCreate(
                [
                    'user_id' => $transaction->user_id,
                    'pattern' => $pattern,
                    'scope' => 'merchant',
                ],
                [
                    'department_id' => $transaction->department_id,
                    'account_id' => $transaction->account_id,
                    'is_personal' => $transaction->is_personal,
                    'weight' => $classification['confidence'] ?? 0.5,
                ]
            );
        }
    }

    /**
     * Get fallback classification using rules
     */
    private function getFallbackClassification(StatementTransaction $transaction): array
    {
        $rule = ClassificationRule::where('scope', 'merchant')
            ->where('pattern', 'like', '%' . addcslashes($transaction->merchant_name, '%_') . '%')
            ->orderByDesc('weight')
            ->first();

        if ($rule) {
            return [
                'success' => true,
                'updates' => [
                    'department' => $rule->department->name ?? null,
                    'account' => $rule->account->name ?? null,
                    'is_personal' => $rule->is_personal,
                ],
                'confidence' => $rule->weight,
                'reasoning' => 'Applied rule-based classification',
            ];
        }

        return [
            'success' => false,
            'updates' => [],
            'confidence' => 0.0,
            'reasoning' => 'No classification rules matched',
        ];
    }

    /**
     * Generate pattern from transaction for rule creation
     */
    private function generatePatternFromTransaction(StatementTransaction $transaction): ?string
    {
        $merchant = strtolower(trim($transaction->merchant_name));
        if (empty($merchant)) {
            return null;
        }

        // Create pattern based on merchant name and key characteristics
        $pattern = $merchant;
        
        // Add MCC if available and significant
        if ($transaction->mcc && $transaction->mcc !== '0000') {
            $pattern .= " mcc:{$transaction->mcc}";
        }

        return $pattern;
    }

    /**
     * Get amount range for pattern matching
     */
    private function getAmountRange(float $amount): string
    {
        if ($amount < 10) return 'small';
        if ($amount < 50) return 'medium';
        if ($amount < 200) return 'large';
        return 'very_large';
    }

    /**
     * Calculate merchant confidence based on name quality
     */
    private function calculateMerchantConfidence(string $merchantName): float
    {
        $confidence = 0.5; // Base confidence
        
        // Increase confidence for longer, more descriptive names
        if (strlen($merchantName) > 10) $confidence += 0.1;
        if (strlen($merchantName) > 20) $confidence += 0.1;
        
        // Decrease confidence for generic names
        $genericPatterns = ['pos', 'terminal', 'payment', 'transaction'];
        foreach ($genericPatterns as $pattern) {
            if (stripos($merchantName, $pattern) !== false) {
                $confidence -= 0.2;
            }
        }
        
        return max(0.1, min(1.0, $confidence));
    }

    /**
     * Get amount significance level
     */
    private function getAmountSignificance(float $amount): string
    {
        if ($amount < 5) return 'trivial';
        if ($amount < 25) return 'small';
        if ($amount < 100) return 'moderate';
        if ($amount < 500) return 'significant';
        return 'major';
    }

    /**
     * Get allowed departments
     */
    private function getAllowedDepartments(): array
    {
        return Department::pluck('name')->all();
    }

    /**
     * Get allowed accounts
     */
    private function getAllowedAccounts(): array
    {
        return Account::pluck('name')->all();
    }

    /**
     * Map department name to ID
     */
    private function mapDepartmentNameToId(?string $name): ?int
    {
        if (!$name) return null;
        return Department::where('name', $name)->value('id');
    }

    /**
     * Map account name to ID
     */
    private function mapAccountNameToId(?string $name): ?int
    {
        if (!$name) return null;
        return Account::where('name', $name)->value('id');
    }
}

