<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class LibraryTrack extends Model
{
    protected $fillable = [
        'category_id',
        'title',
        'slug',
        'description',
        'transcript',
        'duration',
        'file_path',
        'file_size',
        'voice_type',
        'difficulty_level',
        'tags',
        'safety_notes',
        'is_active',
        'play_count',
        'rating',
    ];

    protected $casts = [
        'tags' => 'array',
        'safety_notes' => 'array',
        'is_active' => 'boolean',
        'play_count' => 'integer',
        'rating' => 'decimal:2',
    ];

    public function category(): BelongsTo
    {
        return $this->belongsTo(LibraryCategory::class);
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByDifficulty($query, $level)
    {
        return $query->where('difficulty_level', $level);
    }

    public function scopeByVoiceType($query, $type)
    {
        return $query->where('voice_type', $type);
    }

    public function getFormattedDurationAttribute()
    {
        if (!$this->duration) return null;
        
        $minutes = floor($this->duration / 60);
        $seconds = $this->duration % 60;
        
        return sprintf('%d:%02d', $minutes, $seconds);
    }

    public function getFormattedFileSizeAttribute()
    {
        if (!$this->file_size) return null;
        
        $units = ['B', 'KB', 'MB', 'GB'];
        $size = $this->file_size;
        $unit = 0;
        
        while ($size >= 1024 && $unit < count($units) - 1) {
            $size /= 1024;
            $unit++;
        }
        
        return round($size, 2) . ' ' . $units[$unit];
    }
}
