<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Item extends Model
{
    protected $fillable = [
        'category_id',
        'season_id',
        'name',
        'slug',
        'sku',
        'description',
        'colour_grid_columns',
        'has_colourways',
        'attributes',
        'is_active',
    ];

    protected $casts = [
        'colour_grid_columns' => 'integer',
        'has_colourways' => 'boolean',
        'attributes' => 'array',
        'is_active' => 'boolean',
    ];

    /**
     * Get the category that owns the item.
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    /**
     * Get the season that owns the item.
     */
    public function season(): BelongsTo
    {
        return $this->belongsTo(Season::class);
    }

    /**
     * Get the colourways for the item.
     */
    public function colourways(): HasMany
    {
        return $this->hasMany(Colourway::class);
    }

    /**
     * Get the order lines for the item.
     */
    public function orderLines(): HasMany
    {
        return $this->hasMany(OrderLine::class);
    }

    /**
     * Scope a query to only include active items.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to only include items by category.
     */
    public function scopeByCategory($query, $categoryId)
    {
        return $query->where('category_id', $categoryId);
    }

    /**
     * Scope a query to only include items by season.
     */
    public function scopeBySeason($query, $seasonId)
    {
        return $query->where('season_id', $seasonId);
    }

    /**
     * Scope a query to only include items with colourways.
     */
    public function scopeWithColourways($query)
    {
        return $query->where('has_colourways', true);
    }

    /**
     * Scope a query to only include items without colourways.
     */
    public function scopeWithoutColourways($query)
    {
        return $query->where('has_colourways', false);
    }
}
