<?php

namespace App\Services\AI;

use App\Models\StatementTransaction;
use App\Models\Receipt;
use App\Models\Department;
use App\Models\Account;
use App\Models\ClassificationRule;
use Illuminate\Support\Facades\Log;

class MockExpenseClassificationAgent
{
    /**
     * Mock classification that demonstrates the system working
     */
    public function classifyTransaction(StatementTransaction $transaction, ?Receipt $receipt = null): array
    {
        // Simulate some processing time
        usleep(500000); // 0.5 seconds

        $merchant = strtolower($transaction->merchant_name ?? '');
        $amount = $transaction->amount ?? 0;

        // Mock classification logic based on merchant patterns
        $classification = $this->getMockClassification($merchant, $amount, $receipt);
        
        // Apply classification
        $updates = [];
        if ($classification['department_id']) {
            $transaction->department_id = $classification['department_id'];
            $updates['department'] = $classification['department_name'];
        }
        if ($classification['account_id']) {
            $transaction->account_id = $classification['account_id'];
            $updates['account'] = $classification['account_name'];
        }
        if (isset($classification['is_personal'])) {
            $transaction->is_personal = $classification['is_personal'];
            $updates['is_personal'] = $classification['is_personal'];
        }

        if (!empty($updates)) {
            $transaction->save();
        }

        // Create a learning rule for future use
        $this->createLearningRule($transaction, $classification);

        return [
            'success' => !empty($updates),
            'updates' => $updates,
            'confidence' => $classification['confidence'],
            'reasoning' => $classification['reasoning'],
        ];
    }

    /**
     * Re-classify transaction when receipt is matched
     */
    public function reclassifyWithReceipt(StatementTransaction $transaction, Receipt $receipt): array
    {
        return $this->classifyTransaction($transaction, $receipt);
    }

    /**
     * Check if transaction should be split based on receipt analysis
     */
    public function shouldSplitTransaction(StatementTransaction $transaction, Receipt $receipt): array
    {
        // Mock splitting logic - suggest splitting for certain merchants
        $merchant = strtolower($transaction->merchant_name ?? '');
        $shouldSplit = in_array($merchant, ['tesco', 'sainsburys', 'asda', 'morrisons', 'waitrose']);

        if ($shouldSplit) {
            return [
                'should_split' => true,
                'split_lines' => [
                    [
                        'amount' => $transaction->amount * 0.6,
                        'department' => 'Operations',
                        'account' => 'Office Supplies',
                        'description' => 'Office supplies and stationery',
                        'is_personal' => false,
                    ],
                    [
                        'amount' => $transaction->amount * 0.4,
                        'department' => 'HR',
                        'account' => 'Meals',
                        'description' => 'Staff meals and refreshments',
                        'is_personal' => false,
                    ]
                ],
                'reasoning' => 'Grocery store transaction suggests mixed business expenses that should be split between office supplies and staff meals.',
            ];
        }

        return [
            'should_split' => false,
            'split_lines' => [],
            'reasoning' => 'Transaction appears to be a single business expense type.',
        ];
    }

    /**
     * Get mock classification based on merchant patterns
     */
    private function getMockClassification(string $merchant, float $amount, ?Receipt $receipt = null): array
    {
        $departments = Department::all();
        $accounts = Account::all();

        // Mock classification patterns
        $patterns = [
            'fuel' => ['department' => 'Operations', 'account' => 'Travel', 'confidence' => 0.9],
            'petrol' => ['department' => 'Operations', 'account' => 'Travel', 'confidence' => 0.9],
            'bp' => ['department' => 'Operations', 'account' => 'Travel', 'confidence' => 0.8],
            'shell' => ['department' => 'Operations', 'account' => 'Travel', 'confidence' => 0.8],
            'tesco' => ['department' => 'Operations', 'account' => 'Office Supplies', 'confidence' => 0.7],
            'sainsburys' => ['department' => 'Operations', 'account' => 'Office Supplies', 'confidence' => 0.7],
            'asda' => ['department' => 'Operations', 'account' => 'Office Supplies', 'confidence' => 0.7],
            'amazon' => ['department' => 'IT', 'account' => 'Software', 'confidence' => 0.8],
            'microsoft' => ['department' => 'IT', 'account' => 'Software', 'confidence' => 0.9],
            'google' => ['department' => 'IT', 'account' => 'Software', 'confidence' => 0.9],
            'starbucks' => ['department' => 'HR', 'account' => 'Meals', 'confidence' => 0.6],
            'costa' => ['department' => 'HR', 'account' => 'Meals', 'confidence' => 0.6],
            'pret' => ['department' => 'HR', 'account' => 'Meals', 'confidence' => 0.6],
            'mcdonalds' => ['department' => 'HR', 'account' => 'Meals', 'confidence' => 0.5],
            'subway' => ['department' => 'HR', 'account' => 'Meals', 'confidence' => 0.5],
            'hotel' => ['department' => 'Operations', 'account' => 'Travel', 'confidence' => 0.8],
            'travelodge' => ['department' => 'Operations', 'account' => 'Travel', 'confidence' => 0.8],
            'premier inn' => ['department' => 'Operations', 'account' => 'Travel', 'confidence' => 0.8],
            'train' => ['department' => 'Operations', 'account' => 'Travel', 'confidence' => 0.9],
            'bus' => ['department' => 'Operations', 'account' => 'Travel', 'confidence' => 0.7],
            'uber' => ['department' => 'Operations', 'account' => 'Travel', 'confidence' => 0.8],
            'taxi' => ['department' => 'Operations', 'account' => 'Travel', 'confidence' => 0.7],
        ];

        // Find matching pattern
        $match = null;
        foreach ($patterns as $pattern => $config) {
            if (strpos($merchant, $pattern) !== false) {
                $match = $config;
                break;
            }
        }

        // Default classification
        if (!$match) {
            $match = [
                'department' => 'Operations',
                'account' => 'Other',
                'confidence' => 0.3,
            ];
        }

        // Find department and account IDs
        $department = $departments->firstWhere('name', $match['department']);
        $account = $accounts->firstWhere('name', $match['account']);

        // Adjust confidence based on amount and receipt
        $confidence = $match['confidence'];
        if ($amount > 100) {
            $confidence += 0.1; // Higher confidence for larger amounts
        }
        if ($receipt) {
            $confidence += 0.1; // Higher confidence when receipt is available
        }

        return [
            'department_id' => $department?->id,
            'department_name' => $department?->name,
            'account_id' => $account?->id,
            'account_name' => $account?->name,
            'is_personal' => false,
            'confidence' => min(1.0, $confidence),
            'reasoning' => "Classified as {$match['department']} / {$match['account']} based on merchant pattern matching. " .
                          ($receipt ? "Receipt data available for additional context. " : "") .
                          "Amount: £" . number_format($amount, 2),
        ];
    }

    /**
     * Create a learning rule for future classifications
     */
    private function createLearningRule(StatementTransaction $transaction, array $classification): void
    {
        if (!$transaction->user_id || !$classification['department_id'] || !$classification['account_id']) {
            return;
        }

        $pattern = strtolower(trim($transaction->merchant_name));
        if (empty($pattern)) {
            return;
        }

        ClassificationRule::updateOrCreate(
            [
                'user_id' => $transaction->user_id,
                'pattern' => $pattern,
                'scope' => 'merchant',
            ],
            [
                'department_id' => $classification['department_id'],
                'account_id' => $classification['account_id'],
                'is_personal' => $classification['is_personal'] ?? false,
                'weight' => $classification['confidence'],
            ]
        );
    }
}
