# Database Schema Documentation

## Overview

This document describes the complete database schema for the Laravel Livewire application. The schema is designed to support a fashion/textile business with seasons, categories, items, colourways, customers, and orders.

## Database Design Principles

- **Primary Keys**: All tables use `bigIncrements` for primary keys
- **Foreign Keys**: Proper foreign key constraints with appropriate cascade/restrict rules
- **Data Types**: 
  - `timestampTz` for datetime fields (PostgreSQL timezone support)
  - `jsonb` for flexible attributes (PostgreSQL JSON support)
  - Appropriate string lengths for all text fields
- **Constraints**: PostgreSQL CHECK constraints for data validation
- **Indexes**: Strategic indexing for performance optimization

## Table Structure

### 1. Seasons Table

**Purpose**: Manages fashion seasons (Autumn/Winter, Spring/Summer)

```sql
CREATE TABLE seasons (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    code VARCHAR(10) UNIQUE NOT NULL,           -- e.g., "AW25", "SS26"
    half VARCHAR(2) NOT NULL,                   -- "AW" or "SS"
    year SMALLINT NOT NULL,                     -- e.g., 2025
    label VARCHAR(60) NULL,                     -- e.g., "Autumn/Winter 2025"
    starts_on DATE NULL,                        -- Season start date
    ends_on DATE NULL,                          -- Season end date
    is_active BOOLEAN DEFAULT TRUE,             -- Active season flag
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL
);
```

**Indexes**:
- `code` (unique)
- `(half, year)` (unique composite)
- `is_active`

**PostgreSQL CHECK Constraints**:
- `half IN ('AW', 'SS')`
- `year BETWEEN 2020 AND 2100`

**Relationships**:
- Has many `items`

### 2. Categories Table

**Purpose**: Product categorization system

```sql
CREATE TABLE categories (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(150) NOT NULL,                 -- Category name
    slug VARCHAR(160) UNIQUE NOT NULL,          -- URL-friendly slug
    description TEXT NULL,                      -- Category description
    sort_order INT DEFAULT 0,                   -- Display order
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL
);
```

**Indexes**:
- `slug` (unique)
- `sort_order`

**Relationships**:
- Has many `items`

### 3. Items Table

**Purpose**: Main product catalog

```sql
CREATE TABLE items (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    category_id BIGINT NOT NULL,                -- Foreign key to categories
    season_id BIGINT NULL,                      -- Foreign key to seasons (nullable)
    name VARCHAR(200) NOT NULL,                 -- Product name
    slug VARCHAR(200) UNIQUE NOT NULL,          -- URL-friendly slug
    sku VARCHAR(80) UNIQUE NOT NULL,            -- Stock keeping unit
    description TEXT NULL,                      -- Product description
    colour_grid_columns SMALLINT DEFAULT 3,     -- Colour display grid
    has_colourways BOOLEAN DEFAULT TRUE,        -- Whether item has colour variants
    attributes JSONB NULL,                      -- Flexible product attributes
    is_active BOOLEAN DEFAULT TRUE,             -- Active product flag
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL
);
```

**Indexes**:
- `category_id`
- `season_id`
- `is_active`
- `slug` (unique)
- `sku` (unique)

**PostgreSQL CHECK Constraints**:
- `colour_grid_columns BETWEEN 1 AND 8`

**Relationships**:
- Belongs to `category`
- Belongs to `season` (optional)
- Has many `colourways`
- Has many `order_lines`

### 4. Colourways Table

**Purpose**: Colour variants for items

```sql
CREATE TABLE colourways (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    item_id BIGINT NOT NULL,                    -- Foreign key to items
    name VARCHAR(150) NOT NULL,                 -- Colour name (e.g., "Forest Green")
    colour_code VARCHAR(50) NOT NULL,           -- Supplier/house code (e.g., "C23")
    image_path VARCHAR(255) NULL,               -- Swatch image path
    sort_order INT DEFAULT 0,                   -- Display order
    is_active BOOLEAN DEFAULT TRUE,             -- Active colourway flag
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL
);
```

**Indexes**:
- `item_id`
- `(item_id, sort_order)`
- `(item_id, colour_code)` (unique composite)

**Relationships**:
- Belongs to `item`
- Has many `order_lines`

### 5. Customers Table

**Purpose**: Customer information management

```sql
CREATE TABLE customers (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    first_name VARCHAR(120) NOT NULL,           -- Customer first name
    last_name VARCHAR(120) NOT NULL,            -- Customer last name
    email VARCHAR(191) UNIQUE NOT NULL,         -- Customer email
    phone VARCHAR(50) NULL,                     -- Customer phone
    notes TEXT NULL,                            -- Customer notes
    marketing_opt_in BOOLEAN DEFAULT FALSE,     -- Marketing consent
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL
);
```

**Indexes**:
- `email` (unique)

**Relationships**:
- Has many `addresses`
- Has many `orders`

### 6. Addresses Table

**Purpose**: Customer address book

```sql
CREATE TABLE addresses (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    customer_id BIGINT NOT NULL,                -- Foreign key to customers
    label VARCHAR(100) NULL,                    -- Address label (e.g., "Home", "Work")
    contact_name VARCHAR(200) NULL,             -- Contact person name
    line1 VARCHAR(200) NOT NULL,                -- Address line 1
    line2 VARCHAR(200) NULL,                    -- Address line 2
    city VARCHAR(120) NOT NULL,                 -- City
    region VARCHAR(120) NULL,                    -- State/Region
    postcode VARCHAR(20) NOT NULL,              -- Postal code
    country_code CHAR(2) NOT NULL,              -- ISO country code
    phone VARCHAR(50) NULL,                     -- Address-specific phone
    is_default_billing BOOLEAN DEFAULT FALSE,   -- Default billing address
    is_default_shipping BOOLEAN DEFAULT FALSE,  -- Default shipping address
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL
);
```

**Indexes**:
- `customer_id`
- `(customer_id, is_default_billing)`
- `(customer_id, is_default_shipping)`

**Relationships**:
- Belongs to `customer`

### 7. Orders Table

**Purpose**: Order management with address snapshots

```sql
CREATE TABLE orders (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    order_number VARCHAR(30) UNIQUE NOT NULL,   -- Order reference number
    customer_id BIGINT NOT NULL,                -- Foreign key to customers
    status VARCHAR(30) DEFAULT 'draft',         -- Order status
    notes TEXT NULL,                            -- Order notes
    zoho_invoice_number VARCHAR(100) UNIQUE NULL, -- Zoho Books integration
    currency CHAR(3) DEFAULT 'GBP',             -- Order currency
    totals_estimated_minor BIGINT DEFAULT 0,    -- Estimated total (pence)
    totals_final_minor BIGINT DEFAULT 0,        -- Final total (pence)
    placed_at TIMESTAMPTZ NULL,                 -- Order placement timestamp
    invoice_sent_at TIMESTAMPTZ NULL,           -- Invoice sent timestamp
    fulfilled_at TIMESTAMPTZ NULL,              -- Fulfillment timestamp
    
    -- Billing address snapshot
    bill_name VARCHAR(200) NULL,
    bill_line1 VARCHAR(200) NULL,
    bill_line2 VARCHAR(200) NULL,
    bill_city VARCHAR(120) NULL,
    bill_region VARCHAR(120) NULL,
    bill_postcode VARCHAR(20) NULL,
    bill_country_code CHAR(2) NULL,
    bill_phone VARCHAR(50) NULL,
    
    -- Shipping address snapshot
    ship_name VARCHAR(200) NULL,
    ship_line1 VARCHAR(200) NULL,
    ship_line2 VARCHAR(200) NULL,
    ship_city VARCHAR(120) NULL,
    ship_region VARCHAR(120) NULL,
    ship_postcode VARCHAR(20) NULL,
    ship_country_code CHAR(2) NULL,
    ship_phone VARCHAR(50) NULL,
    
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL
);
```

**Indexes**:
- `order_number` (unique)
- `customer_id`
- `status`
- `placed_at`
- `zoho_invoice_number` (unique)

**PostgreSQL CHECK Constraints**:
- `status IN ('draft', 'estimate_sent', 'in_production', 'weighed', 'invoice_sent', 'fulfilled', 'cancelled')`

**Relationships**:
- Belongs to `customer`
- Has many `order_lines`

### 8. Order Lines Table

**Purpose**: Individual line items within orders

```sql
CREATE TABLE order_lines (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    order_id BIGINT NOT NULL,                   -- Foreign key to orders
    item_id BIGINT NOT NULL,                    -- Foreign key to items
    colourway_id BIGINT NULL,                   -- Foreign key to colourways (nullable)
    quantity INT DEFAULT 1,                     -- Quantity ordered
    estimated_weight_g INT NULL,                -- Estimated weight in grams
    final_weight_g INT NULL,                    -- Final weight in grams
    price_estimated_minor BIGINT DEFAULT 0,     -- Estimated price (pence)
    price_final_minor BIGINT DEFAULT 0,         -- Final price (pence)
    factory_po VARCHAR(50) NULL,                -- Factory purchase order
    attributes JSONB NULL,                      -- Line-specific attributes
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL
);
```

**Indexes**:
- `order_id`
- `(item_id, colourway_id)`

**PostgreSQL CHECK Constraints**:
- `quantity > 0`

**Relationships**:
- Belongs to `order`
- Belongs to `item`
- Belongs to `colourway` (optional)

## Business Rules

### 1. Season Management
- Seasons are identified by half (AW/SS) and year
- Each season has a unique code (e.g., "AW25")
- Seasons can be active/inactive
- Items can optionally belong to a season

### 2. Product Structure
- Items belong to categories and optionally to seasons
- Items can have multiple colourways or none
- Items without colourways (e.g., "Swatch Card") set `has_colourways = false`
- When `has_colourways = true`, order lines must include a colourway_id

### 3. Order Management
- Orders maintain address snapshots for audit safety
- Order lines cascade delete with orders
- Items and colourways are restricted from deletion if referenced in orders
- Prices are stored in minor units (pence) for precision

### 4. Customer Management
- Customers can have multiple addresses
- Addresses can be marked as default for billing/shipping
- Orders capture address snapshots at time of creation

## Data Validation

### PostgreSQL CHECK Constraints
- Season half validation: `half IN ('AW', 'SS')`
- Season year validation: `year BETWEEN 2020 AND 2100`
- Item colour grid validation: `colour_grid_columns BETWEEN 1 AND 8`
- Order status validation: `status IN ('draft', 'estimate_sent', 'in_production', 'weighed', 'invoice_sent', 'fulfilled', 'cancelled')`
- Order line quantity validation: `quantity > 0`

### Unique Constraints
- Season code uniqueness
- Season (half, year) uniqueness
- Category slug uniqueness
- Item slug and SKU uniqueness
- Colourway (item_id, colour_code) uniqueness
- Customer email uniqueness
- Order number uniqueness
- Zoho invoice number uniqueness

## Performance Considerations

### Indexes
- Foreign key columns are indexed for join performance
- Composite indexes on frequently queried combinations
- Status and date indexes for order filtering
- Sort order indexes for display ordering

### JSONB Usage
- Product attributes stored as JSONB for flexibility
- Order line attributes for line-specific data
- PostgreSQL JSONB provides efficient querying and indexing

## Migration Notes

- Migrations include database driver detection
- PostgreSQL CHECK constraints are only applied when using PostgreSQL
- SQLite compatibility maintained for development
- All foreign keys use appropriate cascade/restrict rules
- Timestamps use `timestampTz` for timezone support in PostgreSQL

## Sample Data

The seeder creates:
- 2 seasons (AW25, SS26)
- 3 categories (Knitwear, Accessories, Yarns)
- 3 items with various attributes
- 6 colourways across the items
- 1 sample customer with address
- 1 sample order with order line

## Testing

Test routes are available at:
- `/test/seasons` - View all seasons with items
- `/test/categories` - View all categories with items
- `/test/items` - View all active items with relationships
- `/test/customers` - View all customers with addresses and orders
- `/test/orders` - View all orders with full relationships
