<?php

namespace App\Livewire;

use App\Models\CustomRequest;
use Livewire\Component;
use Livewire\Attributes\Title;
use Illuminate\Support\Facades\Auth;

class CustomSessionForm extends Component
{
    public $topic = '';
    public $description = '';
    public $sessionLength = '15min';
    public $voicePreference = 'neutral';
    public $specialNotes = '';
    public $isSubmitting = false;

    protected $rules = [
        'topic' => 'required|string|min:3|max:100',
        'description' => 'required|string|min:10|max:1000',
        'sessionLength' => 'required|in:15min,30min,45min,60min',
        'voicePreference' => 'required|in:male,female,neutral',
        'specialNotes' => 'nullable|string|max:500',
    ];

    protected $messages = [
        'topic.required' => 'Please provide a topic for your session.',
        'topic.min' => 'Topic must be at least 3 characters long.',
        'topic.max' => 'Topic must not exceed 100 characters.',
        'description.required' => 'Please describe what you\'d like in your session.',
        'description.min' => 'Description must be at least 10 characters long.',
        'description.max' => 'Description must not exceed 1000 characters.',
        'specialNotes.max' => 'Special notes must not exceed 500 characters.',
    ];

    public function mount()
    {
        if (Auth::check()) {
            $user = Auth::user();
            if ($user->profile) {
                $this->voicePreference = $user->profile->voice_preference;
            }
        }
    }

    public function submit()
    {
        $this->validate();

        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $this->isSubmitting = true;

        try {
            CustomRequest::create([
                'user_id' => Auth::id(),
                'topic' => $this->topic,
                'description' => $this->description,
                'session_length' => $this->sessionLength,
                'voice_preference' => $this->voicePreference,
                'special_notes' => $this->specialNotes,
                'status' => 'pending',
            ]);

            // Reset form
            $this->reset(['topic', 'description', 'sessionLength', 'specialNotes']);
            
            // Show success message
            session()->flash('success', 'Your custom session request has been submitted successfully! We\'ll review it and get back to you soon.');

        } catch (\Exception $e) {
            session()->flash('error', 'There was an error submitting your request. Please try again.');
        }

        $this->isSubmitting = false;
    }

    #[Title('Request Custom Session - Vybe')]
    public function render()
    {
        return view('livewire.pages.custom-session-form');
    }
}
