<?php

namespace App\Services\Matching;

use App\Models\Receipt;
use App\Models\Statement;
use App\Models\StatementTransaction;
use App\Models\TransactionMatch;
use App\Jobs\ReclassifyWithReceipt;
use Carbon\Carbon;
use App\Services\AI\VertexClient;

class ReceiptMatchingService
{
    public function __construct(private VertexClient $vertex) {}

    public function attemptMatchForReceipt(Receipt $receipt): void
    {
        // Only process receipts that don't already have a match
        if ($receipt->matches()->exists()) {
            return;
        }

            // Get all open statements and use their actual date ranges plus tolerance
            $statements = Statement::query()
                ->where('status', 'open')
                ->get();

            if ($statements->isEmpty()) {
                return; // No open statements; matching deferred
            }

        // Handle date matching more intelligently
        if (empty($receipt->receipt_date)) {
            // No date available - focus on amount and merchant matching only
            \Log::info('No receipt date available, using amount and merchant matching only', [
                'receipt_id' => $receipt->id,
                'merchant' => $receipt->merchant_name,
                'amount' => $receipt->total_amount
            ]);
            
            $candidates = StatementTransaction::query()
                ->whereIn('statement_id', $statements->pluck('id'))
                ->where('has_receipt', false)
                ->whereBetween('amount', [$receipt->total_amount - 0.01, $receipt->total_amount + 0.01]) // Exact amount match
                ->get();
                
            if ($candidates->isEmpty()) {
                // Try with merchant name similarity if exact amount not found
                $merchantWords = explode(' ', strtoupper($receipt->merchant_name));
                $candidates = StatementTransaction::query()
                    ->whereIn('statement_id', $statements->pluck('id'))
                    ->where('has_receipt', false)
                    ->where(function($query) use ($merchantWords) {
                        foreach ($merchantWords as $word) {
                            if (strlen($word) > 2) {
                                $query->orWhere('merchant_name', 'like', '%' . $word . '%');
                            }
                        }
                    })
                    ->whereBetween('amount', [$receipt->total_amount - 5, $receipt->total_amount + 5]) // Within £5
                    ->get();
            }
        } else {
            // Date available - use date-based matching with tolerance
            $receiptDate = Carbon::parse($receipt->receipt_date);
            $today = Carbon::today();
            
            // If receipt date is today or very recent, it might be a fallback date
            // Use a much wider search range
            if ($receiptDate->isSameDay($today) || $receiptDate->isAfter($today->subDays(2))) {
                $toleranceDays = 30; // Much wider range for suspicious dates
                \Log::info('Using wide date range for potentially incorrect receipt date', [
                    'receipt_id' => $receipt->id,
                    'receipt_date' => $receipt->receipt_date,
                    'tolerance_days' => $toleranceDays
                ]);
            } else {
                $toleranceDays = 7; // Normal tolerance for payment processing delays
            }
            
            $start = $receiptDate->subDays($toleranceDays);
            $end = $receiptDate->addDays($toleranceDays);

            $candidates = StatementTransaction::query()
                ->whereIn('statement_id', $statements->pluck('id'))
                ->whereBetween('transaction_date', [$start->toDateString(), $end->toDateString()])
                ->where('has_receipt', false) // Don't match transactions that already have receipts
                ->get();
        }

        if ($candidates->isEmpty()) {
            // If no candidates found in date range, try a broader search focusing on amount and merchant
            \Log::info('No candidates found in date range, trying broader search', [
                'receipt_id' => $receipt->id,
                'receipt_date' => $receipt->receipt_date
            ]);
            
            $candidates = StatementTransaction::query()
                ->whereIn('statement_id', $statements->pluck('id'))
                ->where('has_receipt', false)
                ->whereBetween('amount', [$receipt->total_amount - 0.01, $receipt->total_amount + 0.01]) // Exact amount match
                ->get();
                
            if ($candidates->isEmpty()) {
                // Try with merchant name similarity
                $merchantWords = explode(' ', strtoupper($receipt->merchant_name));
                $candidates = StatementTransaction::query()
                    ->whereIn('statement_id', $statements->pluck('id'))
                    ->where('has_receipt', false)
                    ->where(function($query) use ($merchantWords) {
                        foreach ($merchantWords as $word) {
                            if (strlen($word) > 2) {
                                $query->orWhere('merchant_name', 'like', '%' . $word . '%');
                            }
                        }
                    })
                    ->whereBetween('amount', [$receipt->total_amount - 5, $receipt->total_amount + 5]) // Within £5
                    ->get();
            }
        }

        // Use LLM agent to intelligently match receipt to best transaction
        $matchResult = $this->intelligentMatchWithLLM($receipt, $candidates);

        if ($matchResult && $matchResult['confidence'] >= 0.3) {
            $bestTxn = $candidates->find($matchResult['transaction_id']);
            
            if ($bestTxn) {
                TransactionMatch::updateOrCreate(
                    [
                        'statement_transaction_id' => $bestTxn->id,
                        'receipt_id' => $receipt->id,
                    ],
                    [
                        'matched_amount' => $receipt->total_amount,
                        'match_method' => 'llm',
                        'amount_confidence' => $matchResult['amount_confidence'],
                        'date_confidence' => $matchResult['date_confidence'],
                        'merchant_confidence' => $matchResult['merchant_confidence'],
                        'explanation' => $matchResult['explanation'],
                    ]
                );
                // Mark transaction as having a receipt
                $bestTxn->has_receipt = true;
                $bestTxn->save();
                
                // Trigger re-classification with receipt information
                ReclassifyWithReceipt::dispatch($bestTxn, $receipt)->delay(now()->addSeconds(10));
                
                return; // Successfully matched
            }
        }

        // Fallback: Rule-based matching for exact amount matches within 7 days
        $exactAmountMatch = $candidates->first(function ($txn) use ($receipt) {
            return abs($txn->amount - $receipt->total_amount) < 0.01 && 
                   abs(Carbon::parse($receipt->receipt_date)->diffInDays(Carbon::parse($txn->transaction_date), false)) <= 7;
        });

        if ($exactAmountMatch) {
            TransactionMatch::updateOrCreate(
                [
                    'statement_transaction_id' => $exactAmountMatch->id,
                    'receipt_id' => $receipt->id,
                ],
                [
                    'matched_amount' => $receipt->total_amount,
                    'match_method' => 'rule',
                    'amount_confidence' => 0.95,
                    'date_confidence' => 0.8,
                    'merchant_confidence' => 0.5,
                    'explanation' => 'Exact amount match within 7 days (rule-based fallback)',
                ]
            );
            // Mark transaction as having a receipt
            $exactAmountMatch->has_receipt = true;
            $exactAmountMatch->save();
            
            // Trigger re-classification with receipt information
            ReclassifyWithReceipt::dispatch($exactAmountMatch, $receipt)->delay(now()->addSeconds(10));
        }
    }

    private function merchantConfidence(?string $a, ?string $b): float
    {
        if (!$a || !$b) return 0.5;
        
        $a = $this->normalizeMerchantName($a);
        $b = $this->normalizeMerchantName($b);
        
        if ($a === $b) return 0.9;
        
        // Check for common merchant variations
        if ($this->areMerchantsEquivalent($a, $b)) {
            return 0.85;
        }
        
        similar_text($a, $b, $pct);
        return max(0.5, $pct / 100);
    }
    
    private function normalizeMerchantName(string $name): string
    {
        $name = strtolower(trim($name));
        
        // Remove common suffixes and prefixes
        $name = preg_replace('/\b(ltd|limited|plc|inc|corp|corporation|co|company|store|stores|shop|shops|restaurant|restaurants|garage|garages|station|stations|service|services|petrol|fuel|supermarket|superstore|market|markets)\b/', '', $name);
        
        // Remove common prefixes
        $name = preg_replace('/\b(mr|mrs|ms|dr|prof)\s+/', '', $name);
        
        // Clean up extra spaces
        $name = preg_replace('/\s+/', ' ', $name);
        $name = trim($name);
        
        return $name;
    }
    
    private function areMerchantsEquivalent(string $a, string $b): bool
    {
        // Common merchant name variations
        $variations = [
            'marks and spencer' => ['m&s', 'marks & spencer', 'marks and spencer'],
            'marks & spencer' => ['m&s', 'marks and spencer', 'marks & spencer'],
            'm&s' => ['marks and spencer', 'marks & spencer', 'm&s'],
            'tesco' => ['tesco stores', 'tesco supermarket', 'tesco plc'],
            'sainsbury' => ['sainsburys', 'sainsburys', 'sainsbury stores'],
            'asda' => ['asda stores', 'asda superstore', 'asda supermarket'],
            'morrisons' => ['morrisons supermarket', 'morrisons stores'],
            'waitrose' => ['waitrose supermarket', 'waitrose stores'],
            'aldi' => ['aldi stores', 'aldi supermarket'],
            'lidl' => ['lidl stores', 'lidl supermarket'],
            'bp' => ['bp petrol', 'bp service', 'bp garage', 'bp station'],
            'shell' => ['shell petrol', 'shell service', 'shell garage', 'shell station'],
            'esso' => ['esso petrol', 'esso service', 'esso garage', 'esso station'],
            'texaco' => ['texaco petrol', 'texaco service', 'texaco garage', 'texaco station'],
            'mcdonalds' => ['mcdonalds', 'mcd', 'mcdonald'],
            'burger king' => ['burger king restaurant', 'bk'],
            'kfc' => ['kentucky fried chicken', 'kfc restaurant'],
            'subway' => ['subway restaurant', 'subway sandwich'],
            'pizza hut' => ['pizza hut restaurant', 'pizza hut delivery'],
            'dominos' => ['dominos', 'dominos pizza', 'dominos pizza'],
            'amazon' => ['amazon uk', 'amazon services', 'amazon marketplace'],
            'ebay' => ['ebay uk', 'ebay marketplace'],
            'paypal' => ['paypal uk', 'paypal services'],
            'starbucks' => ['starbucks coffee', 'starbucks cafe'],
            'costa' => ['costa coffee', 'costa cafe'],
            'nero' => ['caffe nero', 'nero coffee'],
            'pret' => ['pret a manger', 'pret a manger coffee'],
            'greggs' => ['greggs bakery', 'greggs store'],
            'boots' => ['boots pharmacy', 'boots chemist', 'boots stores'],
            'superdrug' => ['superdrug pharmacy', 'superdrug chemist', 'superdrug stores'],
            'h&m' => ['h&m stores', 'h&m fashion', 'h&m clothing'],
            'next' => ['next stores', 'next fashion', 'next clothing'],
            'primark' => ['primark stores', 'primark fashion', 'primark clothing'],
            'zara' => ['zara stores', 'zara fashion', 'zara clothing'],
            'uniqlo' => ['uniqlo stores', 'uniqlo fashion', 'uniqlo clothing'],
        ];
        
        // Check if either name is a variation of the other
        foreach ($variations as $base => $alts) {
            if (in_array($a, $alts) && in_array($b, $alts)) {
                return true;
            }
            if ($a === $base && in_array($b, $alts)) {
                return true;
            }
            if ($b === $base && in_array($a, $alts)) {
                return true;
            }
        }
        
        return false;
    }

        /**
         * Intelligent LLM agent to match receipt to best transaction candidate.
         * Analyzes all details comprehensively to make smart matching decisions.
         */
        private function intelligentMatchWithLLM(Receipt $receipt, $candidates): ?array
        {
            if ($candidates->isEmpty()) return null;

            $model = config('vertex.models.match', 'gemini-2.5-flash-lite');
            $system = 'You are an expert expense management assistant for UK businesses. Your task is to intelligently match receipts to credit card transactions using advanced reasoning.

    MATCHING PRIORITY RULES:
    1. EXACT AMOUNT MATCHES: If amounts match exactly (£X.XX), this is OVERWHELMING evidence of a match, even if dates are far apart or merchant names are different. Receipt dates are often incorrect due to OCR issues.
    2. MERCHANT NAME VARIATIONS: Names often differ between receipts and statements:
       - "MARKS & SPENCER" vs "M&S" vs "MARKS AND SPENCER"
       - "ASDA STORES LTD" vs "ASDA" vs "ASDA SUPERSTORE"
       - "BP SERVICE STATION" vs "BP PETROL" vs "BP"
       - "MCDONALDS RESTAURANT" vs "MCDONALDS" vs "MCD"
       - "AMAZON UK SERVICES" vs "AMAZON" vs "AMZN"
       - "SHELL GARAGE" vs "SHELL" vs "SHELL PETROLEUM"
       - "TESCO STORES" vs "TESCO" vs "TESCO SUPERMARKET"
       - "SAINSBURYS" vs "SAINSBURYS" vs "SAINSBURY STORES"
    3. DATE TOLERANCE: Receipt dates are often incorrect due to OCR failures. Focus on amount and merchant matching:
       - If amount matches exactly AND merchant is similar: MATCH regardless of date difference
       - If amount matches exactly: MATCH even if merchant is different (amount is the strongest indicator)
       - Date discrepancies are common due to OCR errors - don't let them prevent good matches
    4. BUSINESS CONTEXT: Consider that merchant names on receipts vs statements are often completely different but represent the same business
    
    CONFIDENCE SCORING:
    - Amount match (exact): +0.95 confidence (overwhelming evidence - ignore date issues)
    - Amount match (within £0.01): +0.9 confidence  
    - Date within 2-3 days: +0.4 confidence (very high)
    - Date within 4-7 days: +0.3 confidence (high)
    - Date beyond 7 days: +0.0 confidence (ignore date issues for exact amount matches)
    - Merchant name similarity: +0.1 to +0.3 confidence
    - Merchant completely different but amount matches exactly: still +0.95 total confidence
    - Exact amount match = 0.95+ confidence regardless of date or merchant name
    
    OUTPUT JSON:
    - transaction_id: ID of best match (or null if no good match)
    - confidence: overall confidence (0.0-1.0)
    - amount_confidence: how confident you are about amount matching (0.0-1.0)
    - date_confidence: how confident you are about date proximity (0.0-1.0) 
    - merchant_confidence: how confident you are about merchant matching (0.0-1.0)
    - explanation: detailed reasoning for your decision
    
    RETURN A MATCH IF:
    - Overall confidence >= 0.3 (very aggressive to catch exact amount matches)
    - OR exact amount match (regardless of date or merchant name) - confidence should be 0.95+
    - OR very close amount match (£0.01 difference) with any merchant similarity - confidence should be 0.9+
    - OR exact amount match with merchant name similarity (even if dates are further apart) - confidence should be 0.8+
    
    CRITICAL: If amounts match exactly and dates are within 7 days, return very high confidence (0.9+) even if merchant names are completely different. Merchant names on receipts vs statements are often totally unrelated but represent the same business transaction.';

        $payload = [
            'receipt' => [
                'id' => $receipt->id,
                'date' => $receipt->receipt_date,
                'merchant' => $receipt->merchant_name,
                'amount' => $receipt->total_amount,
                'currency' => $receipt->currency,
                'lines' => $receipt->lines->map(fn($line) => [
                    'description' => $line->description,
                    'quantity' => $line->quantity,
                    'unit_price' => $line->unit_price,
                    'line_total' => $line->line_total,
                ])->toArray(),
            ],
            'candidate_transactions' => $candidates->map(fn($t) => [
                'id' => $t->id,
                'transaction_date' => $t->transaction_date?->format('Y-m-d'),
                'posted_date' => $t->posted_date?->format('Y-m-d'),
                'merchant_name' => $t->merchant_name,
                'amount' => $t->amount,
                'currency' => $t->currency,
                'original_amount' => $t->original_amount,
                'original_currency' => $t->original_currency,
                'description' => $t->description,
            ])->values()->all(),
        ];

        $prompt = json_encode($payload, JSON_UNESCAPED_SLASHES);
        $resp = $this->vertex->generate($model, $prompt, $system, ['responseMimeType' => 'application/json']);
        
        if (!isset($resp['json']) || !is_array($resp['json'])) {
            return null;
        }

        $result = $resp['json'];
        
        // Validate the response structure
        if (!isset($result['transaction_id']) || !isset($result['confidence'])) {
            return null;
        }

        return [
            'transaction_id' => $result['transaction_id'],
            'confidence' => (float) $result['confidence'],
            'amount_confidence' => (float) ($result['amount_confidence'] ?? 0),
            'date_confidence' => (float) ($result['date_confidence'] ?? 0),
            'merchant_confidence' => (float) ($result['merchant_confidence'] ?? 0),
            'explanation' => $result['explanation'] ?? 'LLM intelligent matching',
        ];
    }
}
