<?php

namespace App\Services\Matching;

use App\Models\Receipt;
use App\Models\Statement;
use App\Models\StatementTransaction;
use App\Models\TransactionMatch;
use Carbon\Carbon;
use App\Services\AI\VertexClient;

class ReceiptMatchingService
{
    public function __construct(private VertexClient $vertex) {}

    public function attemptMatchForReceipt(Receipt $receipt): void
    {
        // Only process receipts that don't already have a match
        if ($receipt->matches()->exists()) {
            return;
        }

        // Get all open statements that could potentially contain this receipt
        $statements = Statement::query()
            ->where('status', 'open')
            ->whereDate('period_start', '<=', $receipt->receipt_date)
            ->whereDate('period_end', '>=', $receipt->receipt_date)
            ->get();

        if ($statements->isEmpty()) {
            return; // No open statements; matching deferred
        }

        // Get a broader set of candidate transactions across all relevant statements
        $toleranceDays = 7; // Increased tolerance for LLM analysis
        $start = Carbon::parse($receipt->receipt_date)->subDays($toleranceDays);
        $end = Carbon::parse($receipt->receipt_date)->addDays($toleranceDays);

        $candidates = StatementTransaction::query()
            ->whereIn('statement_id', $statements->pluck('id'))
            ->whereBetween('transaction_date', [$start->toDateString(), $end->toDateString()])
            ->where('has_receipt', false) // Don't match transactions that already have receipts
            ->get();

        if ($candidates->isEmpty()) {
            return;
        }

            // Use LLM agent to intelligently match receipt to best transaction
            $matchResult = $this->intelligentMatchWithLLM($receipt, $candidates);

            if ($matchResult && $matchResult['confidence'] >= 0.6) {
            $bestTxn = $candidates->find($matchResult['transaction_id']);
            
            if ($bestTxn) {
                TransactionMatch::updateOrCreate(
                    [
                        'statement_transaction_id' => $bestTxn->id,
                        'receipt_id' => $receipt->id,
                    ],
                    [
                        'matched_amount' => $receipt->total_amount,
                        'match_method' => 'llm',
                        'amount_confidence' => $matchResult['amount_confidence'],
                        'date_confidence' => $matchResult['date_confidence'],
                        'merchant_confidence' => $matchResult['merchant_confidence'],
                        'explanation' => $matchResult['explanation'],
                    ]
                );
                // Mark transaction as having a receipt
                $bestTxn->has_receipt = true;
                $bestTxn->save();
            }
        }
    }

    private function merchantConfidence(?string $a, ?string $b): float
    {
        if (!$a || !$b) return 0.5;
        $a = strtolower(trim($a));
        $b = strtolower(trim($b));
        if ($a === $b) return 0.9;
        similar_text($a, $b, $pct);
        return max(0.5, $pct / 100);
    }

        /**
         * Intelligent LLM agent to match receipt to best transaction candidate.
         * Analyzes all details comprehensively to make smart matching decisions.
         */
        private function intelligentMatchWithLLM(Receipt $receipt, $candidates): ?array
        {
            if ($candidates->isEmpty()) return null;

            $model = config('vertex.models.match', 'gemini-2.5-flash-lite');
            $system = 'You are an expert expense management assistant for UK businesses. Your task is to intelligently match receipts to credit card transactions using advanced reasoning.

    MATCHING PRIORITY RULES:
    1. EXACT AMOUNT MATCHES: If amounts match exactly (£X.XX), this is VERY STRONG evidence of a match, even if merchant names are completely different
    2. MERCHANT NAME VARIATIONS: Names often differ between receipts and statements:
       - "ASDA STORES LTD" vs "ASDA" vs "ASDA SUPERSTORE"
       - "BP SERVICE STATION" vs "BP PETROL" vs "BP"
       - "MCDONALDS RESTAURANT" vs "MCDONALDS" vs "MCD"
       - "AMAZON UK SERVICES" vs "AMAZON" vs "AMZN"
       - "SHELL GARAGE" vs "SHELL" vs "SHELL PETROLEUM"
    3. DATE TOLERANCE: Receipts are usually dated on purchase day, transactions appear 1-7 days later due to processing delays
    4. BUSINESS CONTEXT: Consider that merchant names on receipts vs statements are often completely different but represent the same business
    
    CONFIDENCE SCORING:
    - Amount match (exact): +0.8 confidence
    - Amount match (within £0.01): +0.7 confidence  
    - Date within 3 days: +0.3 confidence
    - Date within 7 days: +0.2 confidence
    - Merchant name similarity: +0.1 to +0.4 confidence
    - Merchant completely different but amount matches: still +0.6 total confidence
    
    OUTPUT JSON:
    - transaction_id: ID of best match (or null if no good match)
    - confidence: overall confidence (0.0-1.0)
    - amount_confidence: how confident you are about amount matching (0.0-1.0)
    - date_confidence: how confident you are about date proximity (0.0-1.0) 
    - merchant_confidence: how confident you are about merchant matching (0.0-1.0)
    - explanation: detailed reasoning for your decision
    
    RETURN A MATCH IF:
    - Overall confidence >= 0.6 (lowered from 0.7)
    - OR exact amount match with date within 7 days (regardless of merchant name)
    - OR very close amount match (£0.01 difference) with reasonable date proximity';

        $payload = [
            'receipt' => [
                'id' => $receipt->id,
                'date' => $receipt->receipt_date,
                'merchant' => $receipt->merchant_name,
                'amount' => $receipt->total_amount,
                'currency' => $receipt->currency,
                'lines' => $receipt->lines->map(fn($line) => [
                    'description' => $line->description,
                    'quantity' => $line->quantity,
                    'unit_price' => $line->unit_price,
                    'line_total' => $line->line_total,
                ])->toArray(),
            ],
            'candidate_transactions' => $candidates->map(fn($t) => [
                'id' => $t->id,
                'transaction_date' => $t->transaction_date?->format('Y-m-d'),
                'posted_date' => $t->posted_date?->format('Y-m-d'),
                'merchant_name' => $t->merchant_name,
                'amount' => $t->amount,
                'currency' => $t->currency,
                'original_amount' => $t->original_amount,
                'original_currency' => $t->original_currency,
                'description' => $t->description,
            ])->values()->all(),
        ];

        $prompt = json_encode($payload, JSON_UNESCAPED_SLASHES);
        $resp = $this->vertex->generate($model, $prompt, $system, ['responseMimeType' => 'application/json']);
        
        if (!isset($resp['json']) || !is_array($resp['json'])) {
            return null;
        }

        $result = $resp['json'];
        
        // Validate the response structure
        if (!isset($result['transaction_id']) || !isset($result['confidence'])) {
            return null;
        }

        return [
            'transaction_id' => $result['transaction_id'],
            'confidence' => (float) $result['confidence'],
            'amount_confidence' => (float) ($result['amount_confidence'] ?? 0),
            'date_confidence' => (float) ($result['date_confidence'] ?? 0),
            'merchant_confidence' => (float) ($result['merchant_confidence'] ?? 0),
            'explanation' => $result['explanation'] ?? 'LLM intelligent matching',
        ];
    }
}


