<?php

namespace App\Services\Matching;

use App\Models\Receipt;
use App\Models\Statement;
use App\Models\StatementTransaction;
use App\Models\TransactionMatch;
use Carbon\Carbon;
use App\Services\AI\VertexClient;

class ReceiptMatchingService
{
    public function __construct(private VertexClient $vertex) {}

    public function attemptMatchForReceipt(Receipt $receipt): void
    {
        // Only process receipts that don't already have a match
        if ($receipt->matches()->exists()) {
            return;
        }

        $statement = Statement::query()
            ->where('status', 'open')
            ->whereDate('period_start', '<=', $receipt->receipt_date)
            ->whereDate('period_end', '>=', $receipt->receipt_date)
            ->first();

        if (!$statement) {
            return; // No open statement; matching deferred
        }

        $toleranceDays = 5;
        $start = Carbon::parse($receipt->receipt_date)->subDays($toleranceDays);
        $end = Carbon::parse($receipt->receipt_date)->addDays($toleranceDays);

        $candidates = StatementTransaction::query()
            ->where('statement_id', $statement->id)
            ->whereBetween('transaction_date', [$start->toDateString(), $end->toDateString()])
            ->where(function ($q) use ($receipt) {
                $q->where('amount', $receipt->total_amount)
                  ->orWhere('original_amount', $receipt->total_amount);
            })
            ->get();

        // Try LLM scoring first
        $llmScores = $this->scoreWithLLM($receipt, $candidates);
        $bestTxn = null;
        $bestScore = -1.0;
        $bestDetails = null;

        if (!empty($llmScores)) {
            foreach ($candidates as $txn) {
                $score = $llmScores[$txn->id] ?? null;
                if (!$score) continue;
                $combined = (
                    (float)($score['amount_confidence'] ?? 0)
                    + (float)($score['date_confidence'] ?? 0)
                    + (float)($score['merchant_confidence'] ?? 0)
                );
                if ($combined > $bestScore) {
                    $bestScore = $combined;
                    $bestTxn = $txn;
                    $bestDetails = [
                        'amount_confidence' => (float)($score['amount_confidence'] ?? 0),
                        'date_confidence' => (float)($score['date_confidence'] ?? 0),
                        'merchant_confidence' => (float)($score['merchant_confidence'] ?? 0),
                        'explanation' => $score['explanation'] ?? 'LLM suggested match.',
                    ];
                }
            }
            // Require an average confidence >= ~0.67 (sum >= 2.0)
            if ($bestScore < 2.0) {
                $bestTxn = null;
                $bestDetails = null;
            }
        }

        // Fallback simple rule if LLM didn't yield a confident match
        if (!$bestTxn) {
            $ruleBest = null;
            $ruleBestScore = -1.0;
            foreach ($candidates as $txn) {
                $amountMatches = (float)$txn->amount === (float)$receipt->total_amount
                    || (float)$txn->original_amount === (float)$receipt->total_amount;
                if (!$amountMatches) continue;
                $mc = $this->merchantConfidence($receipt->merchant_name, $txn->merchant_name);
                if ($mc > $ruleBestScore) {
                    $ruleBestScore = $mc;
                    $ruleBest = $txn;
                }
            }
            if ($ruleBest && $ruleBestScore >= 0.7) {
                $bestTxn = $ruleBest;
                $bestDetails = [
                    'amount_confidence' => 0.9,
                    'date_confidence' => 0.7,
                    'merchant_confidence' => $ruleBestScore,
                    'explanation' => 'Rule-based match by amount and merchant similarity.',
                ];
            }
        }

        if ($bestTxn && $bestDetails) {
            TransactionMatch::updateOrCreate(
                [
                    'statement_transaction_id' => $bestTxn->id,
                    'receipt_id' => $receipt->id,
                ],
                [
                    'matched_amount' => $receipt->total_amount,
                    'match_method' => isset($llmScores[$bestTxn->id]) ? 'llm' : 'rule',
                    'amount_confidence' => $bestDetails['amount_confidence'],
                    'date_confidence' => $bestDetails['date_confidence'],
                    'merchant_confidence' => $bestDetails['merchant_confidence'],
                    'explanation' => $bestDetails['explanation'],
                ]
            );
            // Mark transaction as having a receipt for convenience filters
            $bestTxn->has_receipt = true;
            $bestTxn->save();
        }
    }

    private function merchantConfidence(?string $a, ?string $b): float
    {
        if (!$a || !$b) return 0.5;
        $a = strtolower(trim($a));
        $b = strtolower(trim($b));
        if ($a === $b) return 0.9;
        similar_text($a, $b, $pct);
        return max(0.5, $pct / 100);
    }

    /**
     * Ask Vertex to score candidate transactions for a receipt.
     * Returns map: txnId => [amount_confidence, date_confidence, merchant_confidence, explanation]
     */
    private function scoreWithLLM(Receipt $receipt, $candidates): array
    {
        if ($candidates->isEmpty()) return [];

        $model = config('vertex.models.match', 'gemini-1.5-flash');
        $system = 'You are a finance assistant for UK businesses. Score how likely each credit card transaction matches the provided receipt. Consider date tolerance (processing delay), amount (including currency conversions), and merchant name similarity. Output strict JSON object mapping transaction_id to scores, with fields amount_confidence, date_confidence, merchant_confidence (0..1), and short explanation. No extra text.';
        $payload = [
            'receipt' => [
                'date' => (string) $receipt->receipt_date,
                'merchant' => $receipt->merchant_name,
                'amount' => $receipt->total_amount,
                'currency' => $receipt->currency,
            ],
            'candidates' => $candidates->map(fn($t) => [
                'id' => $t->id,
                'transaction_date' => $t->transaction_date?->format('Y-m-d'),
                'posted_date' => $t->posted_date?->format('Y-m-d'),
                'merchant_name' => $t->merchant_name,
                'amount' => $t->amount,
                'currency' => $t->currency,
                'original_amount' => $t->original_amount,
                'original_currency' => $t->original_currency,
            ])->values()->all(),
        ];

        $prompt = json_encode($payload, JSON_UNESCAPED_SLASHES);
        $resp = $this->vertex->generate($model, $prompt, $system, ['responseMimeType' => 'application/json']);
        $json = $resp['json'];
        if (!is_array($json)) return [];
        $scores = [];
        foreach ($json as $id => $row) {
            $scores[(int)$id] = [
                'amount_confidence' => isset($row['amount_confidence']) ? (float)$row['amount_confidence'] : null,
                'date_confidence' => isset($row['date_confidence']) ? (float)$row['date_confidence'] : null,
                'merchant_confidence' => isset($row['merchant_confidence']) ? (float)$row['merchant_confidence'] : null,
                'explanation' => $row['explanation'] ?? null,
            ];
        }
        return $scores;
    }
}


