<?php

namespace App\Services\Matching;

use App\Models\Receipt;
use App\Models\Statement;
use App\Models\StatementTransaction;
use App\Models\TransactionMatch;
use App\Jobs\ReclassifyWithReceipt;
use Carbon\Carbon;
use App\Services\AI\VertexClient;

class ReceiptMatchingService
{
    public function __construct(private VertexClient $vertex) {}

    public function attemptMatchForReceipt(Receipt $receipt): void
    {
        // Only process receipts that don't already have a match
        if ($receipt->matches()->exists()) {
            return;
        }

            // Get all open statements and use their actual date ranges plus tolerance
            $statements = Statement::query()
                ->where('status', 'open')
                ->get();

            if ($statements->isEmpty()) {
                return; // No open statements; matching deferred
            }

            // Use statement date ranges plus 7 days tolerance for payment processing delays
            $toleranceDays = 7; // Payment processing delay tolerance
            $start = Carbon::parse($receipt->receipt_date)->subDays($toleranceDays);
            $end = Carbon::parse($receipt->receipt_date)->addDays($toleranceDays);

        $candidates = StatementTransaction::query()
            ->whereIn('statement_id', $statements->pluck('id'))
            ->whereBetween('transaction_date', [$start->toDateString(), $end->toDateString()])
            ->where('has_receipt', false) // Don't match transactions that already have receipts
            ->get();

        if ($candidates->isEmpty()) {
            return;
        }

        // Use LLM agent to intelligently match receipt to best transaction
        $matchResult = $this->intelligentMatchWithLLM($receipt, $candidates);

        if ($matchResult && $matchResult['confidence'] >= 0.4) {
            $bestTxn = $candidates->find($matchResult['transaction_id']);
            
            if ($bestTxn) {
                TransactionMatch::updateOrCreate(
                    [
                        'statement_transaction_id' => $bestTxn->id,
                        'receipt_id' => $receipt->id,
                    ],
                    [
                        'matched_amount' => $receipt->total_amount,
                        'match_method' => 'llm',
                        'amount_confidence' => $matchResult['amount_confidence'],
                        'date_confidence' => $matchResult['date_confidence'],
                        'merchant_confidence' => $matchResult['merchant_confidence'],
                        'explanation' => $matchResult['explanation'],
                    ]
                );
                // Mark transaction as having a receipt
                $bestTxn->has_receipt = true;
                $bestTxn->save();
                
                // Trigger re-classification with receipt information
                ReclassifyWithReceipt::dispatch($bestTxn, $receipt)->delay(now()->addSeconds(10));
                
                return; // Successfully matched
            }
        }

        // Fallback: Rule-based matching for exact amount matches within 7 days
        $exactAmountMatch = $candidates->first(function ($txn) use ($receipt) {
            return abs($txn->amount - $receipt->total_amount) < 0.01 && 
                   abs(Carbon::parse($receipt->receipt_date)->diffInDays(Carbon::parse($txn->transaction_date), false)) <= 7;
        });

        if ($exactAmountMatch) {
            TransactionMatch::updateOrCreate(
                [
                    'statement_transaction_id' => $exactAmountMatch->id,
                    'receipt_id' => $receipt->id,
                ],
                [
                    'matched_amount' => $receipt->total_amount,
                    'match_method' => 'rule',
                    'amount_confidence' => 0.95,
                    'date_confidence' => 0.8,
                    'merchant_confidence' => 0.5,
                    'explanation' => 'Exact amount match within 7 days (rule-based fallback)',
                ]
            );
            // Mark transaction as having a receipt
            $exactAmountMatch->has_receipt = true;
            $exactAmountMatch->save();
            
            // Trigger re-classification with receipt information
            ReclassifyWithReceipt::dispatch($exactAmountMatch, $receipt)->delay(now()->addSeconds(10));
        }
    }

    private function merchantConfidence(?string $a, ?string $b): float
    {
        if (!$a || !$b) return 0.5;
        
        $a = $this->normalizeMerchantName($a);
        $b = $this->normalizeMerchantName($b);
        
        if ($a === $b) return 0.9;
        
        // Check for common merchant variations
        if ($this->areMerchantsEquivalent($a, $b)) {
            return 0.85;
        }
        
        similar_text($a, $b, $pct);
        return max(0.5, $pct / 100);
    }
    
    private function normalizeMerchantName(string $name): string
    {
        $name = strtolower(trim($name));
        
        // Remove common suffixes and prefixes
        $name = preg_replace('/\b(ltd|limited|plc|inc|corp|corporation|co|company|store|stores|shop|shops|restaurant|restaurants|garage|garages|station|stations|service|services|petrol|fuel|supermarket|superstore|market|markets)\b/', '', $name);
        
        // Remove common prefixes
        $name = preg_replace('/\b(mr|mrs|ms|dr|prof)\s+/', '', $name);
        
        // Clean up extra spaces
        $name = preg_replace('/\s+/', ' ', $name);
        $name = trim($name);
        
        return $name;
    }
    
    private function areMerchantsEquivalent(string $a, string $b): bool
    {
        // Common merchant name variations
        $variations = [
            'marks and spencer' => ['m&s', 'marks & spencer', 'marks and spencer'],
            'marks & spencer' => ['m&s', 'marks and spencer', 'marks & spencer'],
            'm&s' => ['marks and spencer', 'marks & spencer', 'm&s'],
            'tesco' => ['tesco stores', 'tesco supermarket', 'tesco plc'],
            'sainsbury' => ['sainsburys', 'sainsburys', 'sainsbury stores'],
            'asda' => ['asda stores', 'asda superstore', 'asda supermarket'],
            'morrisons' => ['morrisons supermarket', 'morrisons stores'],
            'waitrose' => ['waitrose supermarket', 'waitrose stores'],
            'aldi' => ['aldi stores', 'aldi supermarket'],
            'lidl' => ['lidl stores', 'lidl supermarket'],
            'bp' => ['bp petrol', 'bp service', 'bp garage', 'bp station'],
            'shell' => ['shell petrol', 'shell service', 'shell garage', 'shell station'],
            'esso' => ['esso petrol', 'esso service', 'esso garage', 'esso station'],
            'texaco' => ['texaco petrol', 'texaco service', 'texaco garage', 'texaco station'],
            'mcdonalds' => ['mcdonalds', 'mcd', 'mcdonald'],
            'burger king' => ['burger king restaurant', 'bk'],
            'kfc' => ['kentucky fried chicken', 'kfc restaurant'],
            'subway' => ['subway restaurant', 'subway sandwich'],
            'pizza hut' => ['pizza hut restaurant', 'pizza hut delivery'],
            'dominos' => ['dominos', 'dominos pizza', 'dominos pizza'],
            'amazon' => ['amazon uk', 'amazon services', 'amazon marketplace'],
            'ebay' => ['ebay uk', 'ebay marketplace'],
            'paypal' => ['paypal uk', 'paypal services'],
            'starbucks' => ['starbucks coffee', 'starbucks cafe'],
            'costa' => ['costa coffee', 'costa cafe'],
            'nero' => ['caffe nero', 'nero coffee'],
            'pret' => ['pret a manger', 'pret a manger coffee'],
            'greggs' => ['greggs bakery', 'greggs store'],
            'boots' => ['boots pharmacy', 'boots chemist', 'boots stores'],
            'superdrug' => ['superdrug pharmacy', 'superdrug chemist', 'superdrug stores'],
            'h&m' => ['h&m stores', 'h&m fashion', 'h&m clothing'],
            'next' => ['next stores', 'next fashion', 'next clothing'],
            'primark' => ['primark stores', 'primark fashion', 'primark clothing'],
            'zara' => ['zara stores', 'zara fashion', 'zara clothing'],
            'uniqlo' => ['uniqlo stores', 'uniqlo fashion', 'uniqlo clothing'],
        ];
        
        // Check if either name is a variation of the other
        foreach ($variations as $base => $alts) {
            if (in_array($a, $alts) && in_array($b, $alts)) {
                return true;
            }
            if ($a === $base && in_array($b, $alts)) {
                return true;
            }
            if ($b === $base && in_array($a, $alts)) {
                return true;
            }
        }
        
        return false;
    }

        /**
         * Intelligent LLM agent to match receipt to best transaction candidate.
         * Analyzes all details comprehensively to make smart matching decisions.
         */
        private function intelligentMatchWithLLM(Receipt $receipt, $candidates): ?array
        {
            if ($candidates->isEmpty()) return null;

            $model = config('vertex.models.match', 'gemini-2.5-flash-lite');
            $system = 'You are an expert expense management assistant for UK businesses. Your task is to intelligently match receipts to credit card transactions using advanced reasoning.

    MATCHING PRIORITY RULES:
    1. EXACT AMOUNT MATCHES: If amounts match exactly (£X.XX), this is VERY STRONG evidence of a match, even if merchant names are completely different
    2. MERCHANT NAME VARIATIONS: Names often differ between receipts and statements:
       - "MARKS & SPENCER" vs "M&S" vs "MARKS AND SPENCER"
       - "ASDA STORES LTD" vs "ASDA" vs "ASDA SUPERSTORE"
       - "BP SERVICE STATION" vs "BP PETROL" vs "BP"
       - "MCDONALDS RESTAURANT" vs "MCDONALDS" vs "MCD"
       - "AMAZON UK SERVICES" vs "AMAZON" vs "AMZN"
       - "SHELL GARAGE" vs "SHELL" vs "SHELL PETROLEUM"
       - "TESCO STORES" vs "TESCO" vs "TESCO SUPERMARKET"
       - "SAINSBURYS" vs "SAINSBURYS" vs "SAINSBURY STORES"
    3. DATE TOLERANCE: Receipts are usually dated on purchase day, transactions appear 1-7 days later due to processing delays
       - Within 2-3 days: Very high confidence (0.9+)
       - Within 4-7 days: High confidence (0.8+)
       - Beyond 7 days: Be more suspicious, require stronger merchant/amount match
    4. BUSINESS CONTEXT: Consider that merchant names on receipts vs statements are often completely different but represent the same business
    
    CONFIDENCE SCORING:
    - Amount match (exact): +0.9 confidence (very strong evidence)
    - Amount match (within £0.01): +0.8 confidence  
    - Date within 2-3 days: +0.4 confidence (very high)
    - Date within 4-7 days: +0.3 confidence (high)
    - Date beyond 7 days: +0.1 confidence (low, be suspicious)
    - Merchant name similarity: +0.1 to +0.3 confidence
    - Merchant completely different but amount matches exactly: still +0.8 total confidence
    - Same date + exact amount = 0.9+ confidence regardless of merchant name
    
    OUTPUT JSON:
    - transaction_id: ID of best match (or null if no good match)
    - confidence: overall confidence (0.0-1.0)
    - amount_confidence: how confident you are about amount matching (0.0-1.0)
    - date_confidence: how confident you are about date proximity (0.0-1.0) 
    - merchant_confidence: how confident you are about merchant matching (0.0-1.0)
    - explanation: detailed reasoning for your decision
    
    RETURN A MATCH IF:
    - Overall confidence >= 0.4 (even more aggressive to catch exact amount matches)
    - OR exact amount match with date within 7 days (regardless of merchant name) - confidence should be 0.9+
    - OR very close amount match (£0.01 difference) with reasonable date proximity - confidence should be 0.8+
    - OR exact amount match with merchant name similarity (even if dates are further apart) - confidence should be 0.8+
    
    CRITICAL: If amounts match exactly and dates are within 7 days, return very high confidence (0.9+) even if merchant names are completely different. Merchant names on receipts vs statements are often totally unrelated but represent the same business transaction.';

        $payload = [
            'receipt' => [
                'id' => $receipt->id,
                'date' => $receipt->receipt_date,
                'merchant' => $receipt->merchant_name,
                'amount' => $receipt->total_amount,
                'currency' => $receipt->currency,
                'lines' => $receipt->lines->map(fn($line) => [
                    'description' => $line->description,
                    'quantity' => $line->quantity,
                    'unit_price' => $line->unit_price,
                    'line_total' => $line->line_total,
                ])->toArray(),
            ],
            'candidate_transactions' => $candidates->map(fn($t) => [
                'id' => $t->id,
                'transaction_date' => $t->transaction_date?->format('Y-m-d'),
                'posted_date' => $t->posted_date?->format('Y-m-d'),
                'merchant_name' => $t->merchant_name,
                'amount' => $t->amount,
                'currency' => $t->currency,
                'original_amount' => $t->original_amount,
                'original_currency' => $t->original_currency,
                'description' => $t->description,
            ])->values()->all(),
        ];

        $prompt = json_encode($payload, JSON_UNESCAPED_SLASHES);
        $resp = $this->vertex->generate($model, $prompt, $system, ['responseMimeType' => 'application/json']);
        
        if (!isset($resp['json']) || !is_array($resp['json'])) {
            return null;
        }

        $result = $resp['json'];
        
        // Validate the response structure
        if (!isset($result['transaction_id']) || !isset($result['confidence'])) {
            return null;
        }

        return [
            'transaction_id' => $result['transaction_id'],
            'confidence' => (float) $result['confidence'],
            'amount_confidence' => (float) ($result['amount_confidence'] ?? 0),
            'date_confidence' => (float) ($result['date_confidence'] ?? 0),
            'merchant_confidence' => (float) ($result['merchant_confidence'] ?? 0),
            'explanation' => $result['explanation'] ?? 'LLM intelligent matching',
        ];
    }
}


