<?php

namespace App\Services\Matching;

use App\Models\Receipt;
use App\Models\Statement;
use App\Models\StatementTransaction;
use App\Models\TransactionMatch;
use Carbon\Carbon;
use App\Services\AI\VertexClient;

class ReceiptMatchingService
{
    public function __construct(private VertexClient $vertex) {}

    public function attemptMatchForReceipt(Receipt $receipt): void
    {
        $statement = Statement::query()
            ->where('status', 'open')
            ->whereDate('period_start', '<=', $receipt->receipt_date)
            ->whereDate('period_end', '>=', $receipt->receipt_date)
            ->first();

        if (!$statement) {
            return; // No open statement; matching deferred
        }

        $toleranceDays = 5;
        $start = Carbon::parse($receipt->receipt_date)->subDays($toleranceDays);
        $end = Carbon::parse($receipt->receipt_date)->addDays($toleranceDays);

        $candidates = StatementTransaction::query()
            ->where('statement_id', $statement->id)
            ->whereBetween('transaction_date', [$start->toDateString(), $end->toDateString()])
            ->where(function ($q) use ($receipt) {
                $q->where('amount', $receipt->total_amount)
                  ->orWhere('original_amount', $receipt->total_amount);
            })
            ->get();

        $llmScores = $this->scoreWithLLM($receipt, $candidates);
        foreach ($candidates as $txn) {
            $score = $llmScores[$txn->id] ?? null;
            $method = $score ? 'llm' : 'rule';
            $amountConfidence = $score['amount_confidence'] ?? 0.7;
            $dateConfidence = $score['date_confidence'] ?? 0.6;
            $merchantConfidence = $score['merchant_confidence'] ?? $this->merchantConfidence($receipt->merchant_name, $txn->merchant_name);
            $explanation = $score['explanation'] ?? 'Rule-based match by amount and date tolerance.';

            TransactionMatch::updateOrCreate(
                [
                    'statement_transaction_id' => $txn->id,
                    'receipt_id' => $receipt->id,
                ],
                [
                    'matched_amount' => $receipt->total_amount,
                    'match_method' => $method,
                    'amount_confidence' => $amountConfidence,
                    'date_confidence' => $dateConfidence,
                    'merchant_confidence' => $merchantConfidence,
                    'explanation' => $explanation,
                ]
            );
        }
    }

    private function merchantConfidence(?string $a, ?string $b): float
    {
        if (!$a || !$b) return 0.5;
        $a = strtolower(trim($a));
        $b = strtolower(trim($b));
        if ($a === $b) return 0.9;
        similar_text($a, $b, $pct);
        return max(0.5, $pct / 100);
    }
}


