<?php

namespace App\Services\Matching;

use App\Models\Receipt;
use App\Models\ReceiptGroup;
use App\Models\Statement;
use App\Models\StatementTransaction;
use App\Models\TransactionMatch;
use App\Jobs\ReclassifyWithReceipt;
use Carbon\Carbon;
use App\Services\AI\VertexClient;

class GroupedReceiptMatchingService
{
    public function __construct(private VertexClient $vertex) {}

    /**
     * Attempt to match a receipt group to a statement transaction
     */
    public function attemptMatchForGroup(ReceiptGroup $group): void
    {
        // Only process groups that don't already have matches
        if ($group->hasMatches()) {
            return;
        }

        // Get all open statements
        $statements = Statement::query()
            ->where('status', 'open')
            ->get();

        if ($statements->isEmpty()) {
            return;
        }

        // Calculate group totals and find candidate transactions
        $groupTotal = $group->total_amount;
        $groupDate = $group->date_range['start'] ?? $group->date_range['end'];
        $primaryMerchant = $group->primary_merchant;

        $candidates = $this->findCandidateTransactions($group, $statements);

        if ($candidates->isEmpty()) {
            \Log::info('No candidate transactions found for receipt group', [
                'group_id' => $group->id,
                'group_total' => $groupTotal,
                'group_date' => $groupDate,
                'primary_merchant' => $primaryMerchant
            ]);
            return;
        }

        // Use LLM to intelligently match the group to the best transaction
        $matchResult = $this->intelligentGroupMatchWithLLM($group, $candidates);

        if ($matchResult && $matchResult['confidence'] >= 0.3) {
            $bestTxn = $candidates->find($matchResult['transaction_id']);
            
            if ($bestTxn) {
                $this->createGroupMatch($group, $bestTxn, $matchResult);
                return;
            }
        }

        // Fallback: Rule-based matching for exact amount matches
        $exactAmountMatch = $candidates->first(function ($txn) use ($groupTotal, $groupDate) {
            return abs($txn->amount - $groupTotal) < 0.01 && 
                   (!$groupDate || abs(Carbon::parse($groupDate)->diffInDays(Carbon::parse($txn->transaction_date), false)) <= 7);
        });

        if ($exactAmountMatch) {
            $this->createGroupMatch($group, $exactAmountMatch, [
                'confidence' => 0.95,
                'amount_confidence' => 0.95,
                'date_confidence' => 0.8,
                'merchant_confidence' => 0.5,
                'explanation' => 'Exact amount match within 7 days (rule-based fallback)',
            ]);
        }
    }

    /**
     * Find candidate transactions for a receipt group
     */
    private function findCandidateTransactions(ReceiptGroup $group, $statements)
    {
        $groupTotal = $group->total_amount;
        $groupDate = $group->date_range['start'] ?? $group->date_range['end'];
        $primaryMerchant = $group->primary_merchant;

        if (!$groupDate) {
            // No date available - focus on amount and merchant matching only
            $candidates = StatementTransaction::query()
                ->whereIn('statement_id', $statements->pluck('id'))
                ->where('has_receipt', false)
                ->whereBetween('amount', [$groupTotal - 0.01, $groupTotal + 0.01])
                ->get();
                
            if ($candidates->isEmpty() && $primaryMerchant) {
                // Try with merchant name similarity
                $merchantWords = explode(' ', strtoupper($primaryMerchant));
                $candidates = StatementTransaction::query()
                    ->whereIn('statement_id', $statements->pluck('id'))
                    ->where('has_receipt', false)
                    ->where(function($query) use ($merchantWords) {
                        foreach ($merchantWords as $word) {
                            if (strlen($word) > 2) {
                                $query->orWhere('merchant_name', 'like', '%' . $word . '%');
                            }
                        }
                    })
                    ->whereBetween('amount', [$groupTotal - 5, $groupTotal + 5])
                    ->get();
            }
        } else {
            // Date available - use date-based matching with tolerance
            $receiptDate = Carbon::parse($groupDate);
            $toleranceDays = 7; // Normal tolerance for payment processing delays
            
            $start = $receiptDate->subDays($toleranceDays);
            $end = $receiptDate->addDays($toleranceDays);

            $candidates = StatementTransaction::query()
                ->whereIn('statement_id', $statements->pluck('id'))
                ->whereBetween('transaction_date', [$start->toDateString(), $end->toDateString()])
                ->where('has_receipt', false)
                ->get();

            if ($candidates->isEmpty()) {
                // Try broader search focusing on amount
                $candidates = StatementTransaction::query()
                    ->whereIn('statement_id', $statements->pluck('id'))
                    ->where('has_receipt', false)
                    ->whereBetween('amount', [$groupTotal - 0.01, $groupTotal + 0.01])
                    ->get();
            }
        }

        return $candidates;
    }

    /**
     * Create a match between a receipt group and a transaction
     */
    private function createGroupMatch(ReceiptGroup $group, StatementTransaction $transaction, array $matchResult): void
    {
        // Create individual matches for each receipt in the group
        foreach ($group->receipts as $receipt) {
            TransactionMatch::updateOrCreate(
                [
                    'statement_transaction_id' => $transaction->id,
                    'receipt_id' => $receipt->id,
                ],
                [
                    'matched_amount' => $receipt->total_amount,
                    'match_method' => 'llm',
                    'amount_confidence' => $matchResult['amount_confidence'],
                    'date_confidence' => $matchResult['date_confidence'],
                    'merchant_confidence' => $matchResult['merchant_confidence'],
                    'explanation' => $matchResult['explanation'] . ' (Grouped receipt)',
                ]
            );
        }

        // Mark transaction as having receipts
        $transaction->has_receipt = true;
        $transaction->save();
        
        // Trigger re-classification with the first receipt (representative)
        $representativeReceipt = $group->receipts->first();
        if ($representativeReceipt) {
            ReclassifyWithReceipt::dispatch($transaction, $representativeReceipt)->delay(now()->addSeconds(10));
        }

        \Log::info('Successfully matched receipt group to transaction', [
            'group_id' => $group->id,
            'transaction_id' => $transaction->id,
            'confidence' => $matchResult['confidence'],
            'receipt_count' => $group->receipts->count()
        ]);
    }

    /**
     * Intelligent LLM agent to match receipt group to best transaction candidate
     */
    private function intelligentGroupMatchWithLLM(ReceiptGroup $group, $candidates): ?array
    {
        if ($candidates->isEmpty()) return null;

        $model = config('vertex.models.match', 'gemini-2.5-flash-lite');
        $system = 'You are an expert expense management assistant for UK businesses. Your task is to intelligently match receipt GROUPS to credit card transactions using advanced reasoning.

A RECEIPT GROUP contains multiple related receipts that together represent one business transaction. Common scenarios:
- Card receipt + itemized receipt from same merchant
- Split receipts (multiple parts of one receipt)
- Related purchases at same location on same day

MATCHING PRIORITY RULES:
1. EXACT AMOUNT MATCHES: If group total matches transaction amount exactly, this is OVERWHELMING evidence
2. MERCHANT NAME VARIATIONS: Names often differ between receipts and statements
3. DATE TOLERANCE: Receipt dates are often incorrect due to OCR failures
4. GROUP CONTEXT: Consider that multiple receipts together make up one transaction

CONFIDENCE SCORING:
- Amount match (exact): +0.95 confidence
- Amount match (within £0.01): +0.9 confidence  
- Date within 2-3 days: +0.4 confidence
- Date within 4-7 days: +0.3 confidence
- Merchant name similarity: +0.1 to +0.3 confidence
- Exact amount match = 0.95+ confidence regardless of date or merchant name

OUTPUT JSON:
- transaction_id: ID of best match (or null if no good match)
- confidence: overall confidence (0.0-1.0)
- amount_confidence: how confident about amount matching (0.0-1.0)
- date_confidence: how confident about date proximity (0.0-1.0) 
- merchant_confidence: how confident about merchant matching (0.0-1.0)
- explanation: detailed reasoning for your decision

RETURN A MATCH IF:
- Overall confidence >= 0.3
- OR exact amount match (regardless of date or merchant name) - confidence should be 0.95+';

        $payload = [
            'receipt_group' => [
                'id' => $group->id,
                'name' => $group->name,
                'total_amount' => $group->total_amount,
                'receipt_count' => $group->receipts->count(),
                'primary_merchant' => $group->primary_merchant,
                'date_range' => $group->date_range,
                'receipts' => $group->receipts->map(fn($receipt) => [
                    'id' => $receipt->id,
                    'date' => $receipt->receipt_date?->format('Y-m-d'),
                    'merchant' => $receipt->merchant_name,
                    'amount' => $receipt->total_amount,
                    'currency' => $receipt->currency,
                    'lines_count' => $receipt->lines->count(),
                ])->toArray(),
            ],
            'candidate_transactions' => $candidates->map(fn($t) => [
                'id' => $t->id,
                'transaction_date' => $t->transaction_date?->format('Y-m-d'),
                'posted_date' => $t->posted_date?->format('Y-m-d'),
                'merchant_name' => $t->merchant_name,
                'amount' => $t->amount,
                'currency' => $t->currency,
                'original_amount' => $t->original_amount,
                'original_currency' => $t->original_currency,
            ])->values()->all(),
        ];

        $prompt = json_encode($payload, JSON_UNESCAPED_SLASHES);
        $resp = $this->vertex->generate($model, $prompt, $system, ['responseMimeType' => 'application/json']);
        
        if (!isset($resp['json']) || !is_array($resp['json'])) {
            return null;
        }

        $result = $resp['json'];
        
        if (!isset($result['transaction_id']) || !isset($result['confidence'])) {
            return null;
        }

        return [
            'transaction_id' => $result['transaction_id'],
            'confidence' => (float) $result['confidence'],
            'amount_confidence' => (float) ($result['amount_confidence'] ?? 0),
            'date_confidence' => (float) ($result['date_confidence'] ?? 0),
            'merchant_confidence' => (float) ($result['merchant_confidence'] ?? 0),
            'explanation' => $result['explanation'] ?? 'LLM intelligent group matching',
        ];
    }
}
