#!/bin/bash

# Multi-Tenant Setup Script for Vybe Photo
# This script helps set up the multi-tenant system

set -e

echo "🎨 Vybe Photo - Multi-Tenant Setup"
echo "===================================="
echo ""

# Check if PostgreSQL is installed
if ! command -v psql &> /dev/null; then
    echo "❌ PostgreSQL is not installed. Please install PostgreSQL first."
    exit 1
fi

echo "✅ PostgreSQL is installed"

# Check environment variables
if [ -z "$DB_NAME" ]; then
    DB_NAME="photowebsite"
    echo "⚠️  DB_NAME not set, using default: $DB_NAME"
fi

if [ -z "$DB_USER" ]; then
    DB_USER="postgres"
    echo "⚠️  DB_USER not set, using default: $DB_USER"
fi

# Install dependencies
echo ""
echo "📦 Installing dependencies..."
npm install

# Ask if user wants to run migrations
echo ""
echo "🔄 Database Migration"
echo "--------------------"
read -p "Do you want to run the multi-tenant migration? (y/n) " -n 1 -r
echo ""

if [[ $REPLY =~ ^[Yy]$ ]]; then
    echo "Running migration..."
    
    if [ -z "$DB_PASSWORD" ]; then
        # Run without password (will prompt)
        psql -U "$DB_USER" -d "$DB_NAME" -f migrations/001-multi-tenant-schema.sql
    else
        # Run with password from environment
        PGPASSWORD="$DB_PASSWORD" psql -U "$DB_USER" -d "$DB_NAME" -f migrations/001-multi-tenant-schema.sql
    fi
    
    echo "✅ Migration completed"
fi

# Ask if user wants to create a default user
echo ""
echo "👤 Create Default User"
echo "--------------------"
read -p "Do you want to create a default photographer account? (y/n) " -n 1 -r
echo ""

if [[ $REPLY =~ ^[Yy]$ ]]; then
    echo ""
    read -p "Username (lowercase, alphanumeric, hyphens): " USERNAME
    read -p "Email: " EMAIL
    read -p "Full Name: " FULLNAME
    read -s -p "Password (min 8 chars): " PASSWORD
    echo ""
    
    # Use Node to hash the password and create the user
    node -e "
    const bcrypt = require('bcrypt');
    const { Pool } = require('pg');
    
    const pool = new Pool({
      user: process.env.DB_USER || 'postgres',
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME || 'photowebsite',
      host: process.env.DB_HOST || 'localhost',
      port: parseInt(process.env.DB_PORT || '5432', 10)
    });
    
    (async () => {
      try {
        const passwordHash = await bcrypt.hash('$PASSWORD', 10);
        const trialEndsAt = new Date(Date.now() + 14 * 24 * 60 * 60 * 1000);
        
        const result = await pool.query(
          \`INSERT INTO users (username, email, password_hash, full_name, subscription_tier, subscription_status, trial_ends_at)
           VALUES (\$1, \$2, \$3, \$4, 'starter', 'trialing', \$5)
           RETURNING id, username, email\`,
          ['$USERNAME', '$EMAIL', passwordHash, '$FULLNAME', trialEndsAt]
        );
        
        console.log('✅ User created successfully!');
        console.log('   ID:', result.rows[0].id);
        console.log('   Username:', result.rows[0].username);
        console.log('   Email:', result.rows[0].email);
        console.log('   Gallery URL: https://$USERNAME.vybephoto.com');
        
        await pool.end();
      } catch (error) {
        console.error('❌ Error creating user:', error.message);
        process.exit(1);
      }
    })();
    "
fi

# Check if there's existing data to migrate
echo ""
echo "📁 Data Migration"
echo "----------------"
read -p "Do you have existing albums/photos to migrate to this user? (y/n) " -n 1 -r
echo ""

if [[ $REPLY =~ ^[Yy]$ ]]; then
    echo ""
    read -p "Enter the user ID to assign data to: " USER_ID
    
    # Migrate existing data
    if [ -z "$DB_PASSWORD" ]; then
        psql -U "$DB_USER" -d "$DB_NAME" <<EOF
-- Assign all albums without user_id to this user
UPDATE albums SET user_id = $USER_ID WHERE user_id IS NULL;

-- Assign all album_groups without user_id to this user
UPDATE album_groups SET user_id = $USER_ID WHERE user_id IS NULL;

-- Update usage statistics
UPDATE users u
SET 
  current_photo_count = (
    SELECT COUNT(*) 
    FROM photos p 
    JOIN albums a ON p.album_id = a.id 
    WHERE a.user_id = u.id
  ),
  current_storage_bytes = (
    SELECT COALESCE(SUM(p.size_bytes), 0)
    FROM photos p
    JOIN albums a ON p.album_id = a.id
    WHERE a.user_id = u.id
  )
WHERE u.id = $USER_ID;

-- Show results
SELECT 
  u.username,
  u.current_photo_count,
  ROUND(u.current_storage_bytes::numeric / 1024 / 1024 / 1024, 2) as storage_gb
FROM users u
WHERE u.id = $USER_ID;
EOF
    else
        PGPASSWORD="$DB_PASSWORD" psql -U "$DB_USER" -d "$DB_NAME" <<EOF
-- Assign all albums without user_id to this user
UPDATE albums SET user_id = $USER_ID WHERE user_id IS NULL;

-- Assign all album_groups without user_id to this user
UPDATE album_groups SET user_id = $USER_ID WHERE user_id IS NULL;

-- Update usage statistics
UPDATE users u
SET 
  current_photo_count = (
    SELECT COUNT(*) 
    FROM photos p 
    JOIN albums a ON p.album_id = a.id 
    WHERE a.user_id = u.id
  ),
  current_storage_bytes = (
    SELECT COALESCE(SUM(p.size_bytes), 0)
    FROM photos p
    JOIN albums a ON p.album_id = a.id
    WHERE a.user_id = u.id
  )
WHERE u.id = $USER_ID;

-- Show results
SELECT 
  u.username,
  u.current_photo_count,
  ROUND(u.current_storage_bytes::numeric / 1024 / 1024 / 1024, 2) as storage_gb
FROM users u
WHERE u.id = $USER_ID;
EOF
    fi
    
    echo "✅ Data migration completed"
fi

# Environment variables reminder
echo ""
echo "🔧 Environment Configuration"
echo "---------------------------"
echo "Make sure to set these environment variables:"
echo ""
echo "  BASE_DOMAIN=vybephoto.com"
echo "  APP_SUBDOMAIN=app"
echo "  USE_POSTGRES=true"
echo ""
echo "For local testing, add to /etc/hosts:"
echo "  127.0.0.1 www.vybephoto.com        # Main site (Joomla)"
echo "  127.0.0.1 app.vybephoto.com        # Login/Dashboard"
echo "  127.0.0.1 app.localhost            # Login/Dashboard (alt)"
echo "  127.0.0.1 $USERNAME.vybephoto.com  # Your gallery"
echo "  127.0.0.1 $USERNAME.localhost      # Your gallery (alt)"
echo ""

# Ask if user wants to start the server
echo ""
read -p "Do you want to start the multi-tenant server now? (y/n) " -n 1 -r
echo ""

if [[ $REPLY =~ ^[Yy]$ ]]; then
    echo ""
    echo "🚀 Starting Vybe Photo multi-tenant server..."
    echo ""
    npm run start
fi

echo ""
echo "✨ Setup complete!"
echo ""
echo "📚 Next steps:"
echo "  1. Configure DNS with wildcard support (*.vybephoto.com)"
echo "  2. Set up SSL/HTTPS for production"
echo "  3. Configure cloud storage (optional)"
echo "  4. Set up Stripe for billing (optional)"
echo ""
echo "📖 See MULTI-TENANT-GUIDE.md for detailed documentation"

