<?php

namespace App\Jobs;

use App\Models\StatementTransaction;
use App\Services\AI\FinanceClassificationAgent;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ClassifyTransaction implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public int $transactionId,
        public bool $hasReceipt = false
    ) {}

    public function handle(FinanceClassificationAgent $agent): void
    {
        $transaction = StatementTransaction::find($this->transactionId);
        
        if (!$transaction) {
            Log::warning('Transaction not found for classification', ['id' => $this->transactionId]);
            return;
        }

        try {
            $result = $agent->classifyTransaction($transaction, $this->hasReceipt);
            
            // Update transaction with classification
            $updates = [];
            
            if ($result['department_id'] && $result['confidence'] >= 0.6) {
                $updates['department_id'] = $result['department_id'];
            }
            
            if ($result['account_id'] && $result['confidence'] >= 0.6) {
                $updates['account_id'] = $result['account_id'];
            }
            
            // Store AI classification metadata
            $meta = is_array($transaction->meta) ? $transaction->meta : (empty($transaction->meta) ? [] : json_decode($transaction->meta, true));
            $meta['ai_classification'] = [
                'department_id' => $result['department_id'],
                'account_id' => $result['account_id'],
                'confidence' => $result['confidence'],
                'reasoning' => $result['reasoning'],
                'timestamp' => now()->toIso8601String(),
                'triggered_by' => $this->hasReceipt ? 'receipt_match' : 'import',
            ];
            
            // Store split suggestions
            if ($result['should_split'] && !empty($result['suggested_splits'])) {
                $meta['suggested_splits'] = $result['suggested_splits'];
            }
            
            $updates['meta'] = $meta;
            
            if (!empty($updates)) {
                $transaction->update($updates);
                Log::info('Transaction classified', [
                    'transaction_id' => $transaction->id,
                    'confidence' => $result['confidence'],
                    'should_split' => $result['should_split']
                ]);
            }
            
        } catch (\Exception $e) {
            Log::error('Transaction classification failed', [
                'transaction_id' => $this->transactionId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Don't fail the job, just log
            return;
        }
    }

    public function failed(\Throwable $exception): void
    {
        Log::error('ClassifyTransaction job failed', [
            'transaction_id' => $this->transactionId,
            'error' => $exception->getMessage()
        ]);
    }
}

