<?php

use Illuminate\Cache\RateLimiting\Limit;
use Illuminate\Http\Request;
use Illuminate\Foundation\Application;
use Illuminate\Foundation\Configuration\Exceptions;
use Illuminate\Foundation\Configuration\Middleware;
use Illuminate\Support\Facades\RateLimiter;

return Application::configure(basePath: dirname(__DIR__))
    ->withRouting(
        web: __DIR__.'/../routes/web.php',
        api: __DIR__.'/../routes/api.php',
        commands: __DIR__.'/../routes/console.php',
        health: '/healthz',
        apiPrefix: 'api',
        then: function () {
            RateLimiter::for('api', function ($request) {
                $key = optional($request->user())->id ?: $request->ip();

                return [Limit::perMinute(120)->by($key)];
            });

            RateLimiter::for('custom-requests', function (Request $request) {
                $key = optional($request->user())->id ?: $request->ip();
                return [
                    Limit::perMinute((int) config('rate_limits.custom_requests.per_minute', 1))->by($key),
                    Limit::perDay((int) config('rate_limits.custom_requests.per_day', 3))->by($key),
                ];
            });
        }
    )
    ->withMiddleware(function (Middleware $middleware) {
        // Avoid redirecting guests to a non-existent login route; return 401 for unauthenticated
        $middleware->redirectGuestsTo(fn () => null);

        $middleware->group('api', [
            \Illuminate\Routing\Middleware\SubstituteBindings::class,
            \Illuminate\Routing\Middleware\ThrottleRequests::class.':api',
            App\Http\Middleware\IdempotencyMiddleware::class,
        ]);

        $middleware->alias([
            'auth.sanctum' => \Laravel\Sanctum\Http\Middleware\CheckAbilities::class,
        ]);
    })
    ->withExceptions(function (Exceptions $exceptions) {
        $exceptions->render(function (Throwable $e) {
            $status = method_exists($e, 'getStatusCode') ? $e->getStatusCode() : 500;
            $traceId = (string) str()->uuid();
            $problem = [
                'type' => 'about:blank',
                'title' => class_basename($e),
                'status' => $status,
                'detail' => $e->getMessage(),
                'trace_id' => $traceId,
            ];
            if (config('app.debug')) {
                $problem['exception'] = [
                    'class' => get_class($e),
                ];
            }

            return response()->json($problem, $status, ['Content-Type' => 'application/problem+json']);
        });
    })->create();
