#!/usr/bin/env node
/**
 * Test Script: Disabled Tiers Functionality
 * 
 * This script tests that disabled tiers work correctly:
 * 1. Existing users can continue using disabled tiers
 * 2. New users cannot sign up for disabled tiers
 * 3. Users cannot upgrade/downgrade to disabled tiers
 * 
 * Usage:
 *   node test-disabled-tiers.js
 */

const stripeService = require('./src/stripe-service');
const db = require('./src/db-multi-tenant');

// Colors for console output
const colors = {
  reset: '\x1b[0m',
  green: '\x1b[32m',
  red: '\x1b[31m',
  yellow: '\x1b[33m',
  blue: '\x1b[34m',
  gray: '\x1b[90m',
};

function log(message, color = 'reset') {
  console.log(`${colors[color]}${message}${colors.reset}`);
}

function success(message) {
  log(`✓ ${message}`, 'green');
}

function error(message) {
  log(`✗ ${message}`, 'red');
}

function info(message) {
  log(`ℹ ${message}`, 'blue');
}

function warn(message) {
  log(`⚠ ${message}`, 'yellow');
}

async function testGetActiveTiers() {
  info('\n1. Testing getSubscriptionTiers(true) - should return only active tiers');
  
  try {
    const activeTiers = await stripeService.getSubscriptionTiers(true);
    const tierKeys = Object.keys(activeTiers);
    
    log(`   Found ${tierKeys.length} active tiers: ${tierKeys.join(', ')}`, 'gray');
    
    let allActive = true;
    for (const [key, tier] of Object.entries(activeTiers)) {
      if (tier.isActive === false) {
        error(`   Tier ${key} is inactive but was returned`);
        allActive = false;
      }
    }
    
    if (allActive) {
      success('All returned tiers are active');
    }
    
    return activeTiers;
  } catch (err) {
    error(`Failed to get active tiers: ${err.message}`);
    throw err;
  }
}

async function testGetAllTiers() {
  info('\n2. Testing getSubscriptionTiers(false) - should return all tiers');
  
  try {
    const allTiers = await stripeService.getSubscriptionTiers(false);
    const tierKeys = Object.keys(allTiers);
    
    log(`   Found ${tierKeys.length} total tiers: ${tierKeys.join(', ')}`, 'gray');
    
    const inactiveTiers = [];
    for (const [key, tier] of Object.entries(allTiers)) {
      if (tier.isActive === false) {
        inactiveTiers.push(key);
      }
    }
    
    if (inactiveTiers.length > 0) {
      warn(`   Found ${inactiveTiers.length} inactive tier(s): ${inactiveTiers.join(', ')}`);
    } else {
      info('   All tiers are currently active');
    }
    
    success('Successfully retrieved all tiers');
    return { allTiers, inactiveTiers };
  } catch (err) {
    error(`Failed to get all tiers: ${err.message}`);
    throw err;
  }
}

async function testGetTierConfig() {
  info('\n3. Testing getTierConfig() - should return config even for disabled tiers');
  
  try {
    const { inactiveTiers } = await testGetAllTiers();
    
    if (inactiveTiers.length === 0) {
      warn('   No inactive tiers to test - skipping this test');
      warn('   To fully test, disable a tier in the superuser dashboard first');
      return;
    }
    
    const testTier = inactiveTiers[0];
    log(`   Testing with inactive tier: ${testTier}`, 'gray');
    
    const tierConfig = await stripeService.getTierConfig(testTier);
    
    if (!tierConfig) {
      error(`   Failed to get config for disabled tier: ${testTier}`);
      return;
    }
    
    log(`   Retrieved config: ${tierConfig.name}, ${tierConfig.maxPhotos} photos, £${(tierConfig.price/100).toFixed(2)}/mo`, 'gray');
    
    if (tierConfig.isActive === false) {
      success('getTierConfig correctly returns disabled tier config');
    } else {
      error('Tier should be marked as inactive');
    }
  } catch (err) {
    error(`Failed to get tier config: ${err.message}`);
  }
}

async function testCreateSubscriptionWithDisabledTier() {
  info('\n4. Testing createSubscription() - should reject disabled tiers');
  
  try {
    const { inactiveTiers } = await testGetAllTiers();
    
    if (inactiveTiers.length === 0) {
      warn('   No inactive tiers to test - skipping this test');
      return;
    }
    
    const testTier = inactiveTiers[0];
    log(`   Attempting to create subscription for disabled tier: ${testTier}`, 'gray');
    
    try {
      // This should fail
      await stripeService.createSubscription('cus_test123', testTier, 0);
      error('   createSubscription should have thrown an error for disabled tier');
    } catch (err) {
      if (err.message.includes('not available') || err.message.includes('inactive')) {
        success('createSubscription correctly rejects disabled tier');
        log(`   Error message: "${err.message}"`, 'gray');
      } else {
        warn(`   Got error but unexpected message: ${err.message}`);
      }
    }
  } catch (err) {
    error(`Test failed: ${err.message}`);
  }
}

async function testUpdateSubscriptionWithDisabledTier() {
  info('\n5. Testing updateSubscription() - should reject disabled tiers');
  
  try {
    const { inactiveTiers } = await testGetAllTiers();
    
    if (inactiveTiers.length === 0) {
      warn('   No inactive tiers to test - skipping this test');
      return;
    }
    
    const testTier = inactiveTiers[0];
    log(`   Attempting to update subscription to disabled tier: ${testTier}`, 'gray');
    
    try {
      // This should fail
      await stripeService.updateSubscription('sub_test123', testTier);
      error('   updateSubscription should have thrown an error for disabled tier');
    } catch (err) {
      if (err.message.includes('not available') || err.message.includes('inactive')) {
        success('updateSubscription correctly rejects disabled tier');
        log(`   Error message: "${err.message}"`, 'gray');
      } else {
        warn(`   Got error but unexpected message: ${err.message}`);
      }
    }
  } catch (err) {
    error(`Test failed: ${err.message}`);
  }
}

async function testCheckoutWithDisabledTier() {
  info('\n6. Testing createCheckoutSession() - should reject disabled tiers');
  
  try {
    const { inactiveTiers } = await testGetAllTiers();
    
    if (inactiveTiers.length === 0) {
      warn('   No inactive tiers to test - skipping this test');
      return;
    }
    
    const testTier = inactiveTiers[0];
    log(`   Attempting to create checkout for disabled tier: ${testTier}`, 'gray');
    
    try {
      // This should fail
      await stripeService.createCheckoutSession(
        'cus_test123',
        testTier,
        'https://example.com/success',
        'https://example.com/cancel'
      );
      error('   createCheckoutSession should have thrown an error for disabled tier');
    } catch (err) {
      if (err.message.includes('not available') || err.message.includes('inactive')) {
        success('createCheckoutSession correctly rejects disabled tier');
        log(`   Error message: "${err.message}"`, 'gray');
      } else {
        warn(`   Got error but unexpected message: ${err.message}`);
      }
    }
  } catch (err) {
    error(`Test failed: ${err.message}`);
  }
}

async function testDatabaseQuery() {
  info('\n7. Testing database getAllSubscriptionTiers() function');
  
  try {
    log('   Testing with activeOnly = true', 'gray');
    const activeTiers = await db.getAllSubscriptionTiers(true);
    log(`   Found ${activeTiers.length} active tiers`, 'gray');
    
    log('   Testing with activeOnly = false', 'gray');
    const allTiers = await db.getAllSubscriptionTiers(false);
    log(`   Found ${allTiers.length} total tiers`, 'gray');
    
    if (allTiers.length >= activeTiers.length) {
      success('Database queries work correctly');
      
      const inactiveCount = allTiers.length - activeTiers.length;
      if (inactiveCount > 0) {
        warn(`   ${inactiveCount} tier(s) are currently disabled`);
      }
    } else {
      error('   Unexpected result: more active tiers than total tiers');
    }
  } catch (err) {
    error(`Database query failed: ${err.message}`);
  }
}

async function runAllTests() {
  log('\n' + '='.repeat(60), 'blue');
  log('DISABLED TIERS FUNCTIONALITY TEST SUITE', 'blue');
  log('='.repeat(60) + '\n', 'blue');
  
  try {
    // Run all tests
    await testGetActiveTiers();
    await testGetAllTiers();
    await testGetTierConfig();
    await testCreateSubscriptionWithDisabledTier();
    await testUpdateSubscriptionWithDisabledTier();
    await testCheckoutWithDisabledTier();
    await testDatabaseQuery();
    
    log('\n' + '='.repeat(60), 'blue');
    log('TEST SUITE COMPLETE', 'blue');
    log('='.repeat(60) + '\n', 'blue');
    
    info('Summary:');
    info('✓ Active tiers are correctly filtered');
    info('✓ All tiers can be retrieved when needed');
    info('✓ Disabled tiers are protected from new signups');
    info('✓ Existing users can still access disabled tier configs');
    
    warn('\nNote: To fully test disabled tier functionality:');
    warn('1. Log in as superuser');
    warn('2. Disable a subscription tier');
    warn('3. Run this test script again');
    warn('4. Verify that existing users on that tier can still use it');
    
  } catch (err) {
    log('\n' + '='.repeat(60), 'red');
    log('TEST SUITE FAILED', 'red');
    log('='.repeat(60) + '\n', 'red');
    error(`Error: ${err.message}`);
    process.exit(1);
  }
  
  // Close database connection
  if (db.pool) {
    await db.pool.end();
  }
}

// Run tests
runAllTests().catch(err => {
  error(`Fatal error: ${err.message}`);
  console.error(err);
  process.exit(1);
});


