<?php

namespace App\Http\Middleware;

use App\Models\IdempotencyKey;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class IdempotencyMiddleware
{
    public function handle(Request $request, Closure $next)
    {
        if ($request->isMethod('get') || $request->isMethod('head')) {
            return $next($request);
        }

        $key = $request->header('Idempotency-Key');
        if (!$key) {
            return $next($request);
        }

        $userId = optional($request->user())->id;
        $requestHash = hash('sha256', $request->getContent());

        $existing = IdempotencyKey::where(['key' => $key, 'user_id' => $userId, 'method' => $request->method(), 'path' => $request->path()])->first();
        if ($existing && $existing->request_hash === $requestHash && $existing->response_code) {
            return response($existing->response_body, (int)$existing->response_code, ['Content-Type' => 'application/json']);
        }

        return DB::transaction(function () use ($request, $next, $key, $userId, $requestHash) {
            $entry = IdempotencyKey::firstOrCreate([
                'key' => $key,
                'user_id' => $userId,
                'method' => $request->method(),
                'path' => $request->path(),
            ], [
                'request_hash' => $requestHash,
            ]);

            $response = $next($request);

            $entry->update([
                'request_hash' => $requestHash,
                'response_code' => $response->getStatusCode(),
                'response_body' => $response->getContent(),
            ]);

            return $response;
        });
    }
}


