-- Migration: Add Multi-Tenant Support
-- This migration adds user accounts and converts the system to multi-tenant

-- Create users table (photographers)
CREATE TABLE IF NOT EXISTS users (
  id SERIAL PRIMARY KEY,
  username TEXT NOT NULL UNIQUE,
  email TEXT NOT NULL UNIQUE,
  password_hash TEXT NOT NULL,
  full_name TEXT,
  
  -- Subscription details
  subscription_tier TEXT NOT NULL DEFAULT 'starter' CHECK (subscription_tier IN ('starter', 'professional', 'business')),
  subscription_status TEXT NOT NULL DEFAULT 'trialing' CHECK (subscription_status IN ('trialing', 'active', 'past_due', 'canceled', 'paused')),
  trial_ends_at TIMESTAMP,
  subscription_ends_at TIMESTAMP,
  stripe_customer_id TEXT UNIQUE,
  stripe_subscription_id TEXT UNIQUE,
  
  -- Plan limits (cached from subscription tier)
  max_photos INTEGER NOT NULL DEFAULT 1000,
  max_storage_bytes BIGINT NOT NULL DEFAULT 5368709120, -- 5GB default
  
  -- Current usage
  current_photo_count INTEGER NOT NULL DEFAULT 0,
  current_storage_bytes BIGINT NOT NULL DEFAULT 0,
  
  -- Branding (per-user configuration)
  site_name TEXT,
  primary_color TEXT DEFAULT '#3b82f6',
  logo_url TEXT,
  hot_chocolate_default_url TEXT,
  hot_chocolate_text TEXT,
  contact_email TEXT,
  social_links JSONB DEFAULT '{}',
  
  -- Custom domain support
  custom_domain TEXT UNIQUE,
  custom_domain_verified BOOLEAN DEFAULT FALSE,
  
  -- Account metadata
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  last_login_at TIMESTAMP,
  
  -- Settings
  allow_downloads BOOLEAN DEFAULT TRUE,
  require_age_verification BOOLEAN DEFAULT FALSE
);

-- Create indexes for users table
CREATE INDEX IF NOT EXISTS idx_users_username ON users(username);
CREATE INDEX IF NOT EXISTS idx_users_email ON users(email);
CREATE INDEX IF NOT EXISTS idx_users_custom_domain ON users(custom_domain);
CREATE INDEX IF NOT EXISTS idx_users_stripe_customer_id ON users(stripe_customer_id);
CREATE INDEX IF NOT EXISTS idx_users_subscription_status ON users(subscription_status);

-- Add user_id to existing tables
ALTER TABLE album_groups ADD COLUMN IF NOT EXISTS user_id INTEGER REFERENCES users(id) ON DELETE CASCADE;
ALTER TABLE albums ADD COLUMN IF NOT EXISTS user_id INTEGER REFERENCES users(id) ON DELETE CASCADE;

-- Create indexes for user_id foreign keys
CREATE INDEX IF NOT EXISTS idx_album_groups_user_id ON album_groups(user_id);
CREATE INDEX IF NOT EXISTS idx_albums_user_id ON albums(user_id);

-- Create sessions table for user authentication
CREATE TABLE IF NOT EXISTS sessions (
  id SERIAL PRIMARY KEY,
  session_id TEXT NOT NULL UNIQUE,
  user_id INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE,
  expires_at TIMESTAMP NOT NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  last_activity_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  user_agent TEXT,
  ip_address TEXT
);

CREATE INDEX IF NOT EXISTS idx_sessions_session_id ON sessions(session_id);
CREATE INDEX IF NOT EXISTS idx_sessions_user_id ON sessions(user_id);
CREATE INDEX IF NOT EXISTS idx_sessions_expires_at ON sessions(expires_at);

-- Create usage_logs table for tracking photo uploads/deletions
CREATE TABLE IF NOT EXISTS usage_logs (
  id SERIAL PRIMARY KEY,
  user_id INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE,
  action TEXT NOT NULL CHECK (action IN ('photo_uploaded', 'photo_deleted', 'storage_updated')),
  photo_count_delta INTEGER DEFAULT 0,
  storage_bytes_delta BIGINT DEFAULT 0,
  metadata JSONB,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX IF NOT EXISTS idx_usage_logs_user_id ON usage_logs(user_id);
CREATE INDEX IF NOT EXISTS idx_usage_logs_created_at ON usage_logs(created_at);

-- Create billing_events table for Stripe webhooks
CREATE TABLE IF NOT EXISTS billing_events (
  id SERIAL PRIMARY KEY,
  user_id INTEGER REFERENCES users(id) ON DELETE SET NULL,
  stripe_event_id TEXT NOT NULL UNIQUE,
  event_type TEXT NOT NULL,
  event_data JSONB NOT NULL,
  processed BOOLEAN DEFAULT FALSE,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX IF NOT EXISTS idx_billing_events_user_id ON billing_events(user_id);
CREATE INDEX IF NOT EXISTS idx_billing_events_stripe_event_id ON billing_events(stripe_event_id);
CREATE INDEX IF NOT EXISTS idx_billing_events_processed ON billing_events(processed);

-- Update existing site_config to be deprecated (users table now holds per-user config)
-- We'll keep it for backwards compatibility but it won't be used in multi-tenant mode
COMMENT ON TABLE site_config IS 'DEPRECATED: Configuration is now stored per-user in users table';

-- Function to update user usage statistics
CREATE OR REPLACE FUNCTION update_user_usage()
RETURNS TRIGGER AS $$
BEGIN
  IF TG_OP = 'INSERT' THEN
    -- Photo added
    UPDATE users 
    SET current_photo_count = current_photo_count + 1,
        current_storage_bytes = current_storage_bytes + COALESCE(NEW.size_bytes, 0),
        updated_at = CURRENT_TIMESTAMP
    WHERE id = (SELECT user_id FROM albums WHERE id = NEW.album_id);
    
    -- Log the usage change
    INSERT INTO usage_logs (user_id, action, photo_count_delta, storage_bytes_delta, metadata)
    SELECT user_id, 'photo_uploaded', 1, COALESCE(NEW.size_bytes, 0), 
           jsonb_build_object('photo_id', NEW.id, 'album_id', NEW.album_id, 'filename', NEW.filename)
    FROM albums WHERE id = NEW.album_id;
    
  ELSIF TG_OP = 'DELETE' THEN
    -- Photo deleted
    UPDATE users 
    SET current_photo_count = GREATEST(current_photo_count - 1, 0),
        current_storage_bytes = GREATEST(current_storage_bytes - COALESCE(OLD.size_bytes, 0), 0),
        updated_at = CURRENT_TIMESTAMP
    WHERE id = (SELECT user_id FROM albums WHERE id = OLD.album_id);
    
    -- Log the usage change
    INSERT INTO usage_logs (user_id, action, photo_count_delta, storage_bytes_delta, metadata)
    SELECT user_id, 'photo_deleted', -1, -COALESCE(OLD.size_bytes, 0),
           jsonb_build_object('photo_id', OLD.id, 'album_id', OLD.album_id, 'filename', OLD.filename)
    FROM albums WHERE id = OLD.album_id;
  END IF;
  
  RETURN NULL;
END;
$$ LANGUAGE plpgsql;

-- Create triggers for automatic usage tracking
DROP TRIGGER IF EXISTS trigger_update_user_usage ON photos;
CREATE TRIGGER trigger_update_user_usage
AFTER INSERT OR DELETE ON photos
FOR EACH ROW EXECUTE FUNCTION update_user_usage();

-- Function to apply subscription tier limits
CREATE OR REPLACE FUNCTION apply_subscription_limits()
RETURNS TRIGGER AS $$
BEGIN
  -- Set limits based on subscription tier
  IF NEW.subscription_tier = 'starter' THEN
    NEW.max_photos := 1000;
    NEW.max_storage_bytes := 5368709120; -- 5GB
  ELSIF NEW.subscription_tier = 'professional' THEN
    NEW.max_photos := 10000;
    NEW.max_storage_bytes := 53687091200; -- 50GB
  ELSIF NEW.subscription_tier = 'business' THEN
    NEW.max_photos := 30000;
    NEW.max_storage_bytes := 161061273600; -- 150GB
  END IF;
  
  NEW.updated_at := CURRENT_TIMESTAMP;
  RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Create trigger to auto-set limits when tier changes
DROP TRIGGER IF EXISTS trigger_apply_subscription_limits ON users;
CREATE TRIGGER trigger_apply_subscription_limits
BEFORE INSERT OR UPDATE OF subscription_tier ON users
FOR EACH ROW EXECUTE FUNCTION apply_subscription_limits();

-- Grant necessary permissions (adjust as needed for your deployment)
-- These are examples; adjust based on your database user setup
-- GRANT SELECT, INSERT, UPDATE, DELETE ON ALL TABLES IN SCHEMA public TO your_app_user;
-- GRANT USAGE, SELECT ON ALL SEQUENCES IN SCHEMA public TO your_app_user;

COMMENT ON TABLE users IS 'Photographer accounts with subscription and branding info';
COMMENT ON TABLE sessions IS 'User authentication sessions';
COMMENT ON TABLE usage_logs IS 'Audit log for photo uploads/deletions and storage changes';
COMMENT ON TABLE billing_events IS 'Stripe webhook events for billing';

