<?php

namespace Tests\Unit;

use Tests\TestCase;
use App\Services\PriceResolutionService;
use App\Services\TotalCacheService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Mockery;

class CacheInvalidationLogicTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test that PriceResolutionService has all required invalidation methods
     */
    public function test_price_resolution_service_has_required_methods()
    {
        $service = app(PriceResolutionService::class);
        
        // Test that all required invalidation methods exist
        $this->assertTrue(method_exists($service, 'invalidateByColourway'));
        $this->assertTrue(method_exists($service, 'invalidateByCustomer'));
        $this->assertTrue(method_exists($service, 'invalidateBySupplier'));
        $this->assertTrue(method_exists($service, 'invalidateByPrice'));
        $this->assertTrue(method_exists($service, 'invalidateByStyle'));
        $this->assertTrue(method_exists($service, 'invalidateByDesign'));
        $this->assertTrue(method_exists($service, 'invalidateByStyleVersion'));
        
        // Test that warmup methods exist
        $this->assertTrue(method_exists($service, 'invalidateAndWarmupByColourway'));
        $this->assertTrue(method_exists($service, 'invalidateAndWarmupByCustomer'));
        $this->assertTrue(method_exists($service, 'invalidateAndWarmupBySupplier'));
        $this->assertTrue(method_exists($service, 'invalidateAndWarmupByPrice'));
        $this->assertTrue(method_exists($service, 'invalidateAndWarmupByDesign'));
        $this->assertTrue(method_exists($service, 'invalidateAndWarmupByStyleVersion'));
        
        // Test that utility methods exist
        $this->assertTrue(method_exists($service, 'warmupColourway'));
        $this->assertTrue(method_exists($service, 'getCacheStats'));
        $this->assertTrue(method_exists($service, 'cleanupStaleResolutions'));
    }

    /**
     * Test that TotalCacheService has all required methods
     */
    public function test_total_cache_service_has_required_methods()
    {
        $service = app(TotalCacheService::class);
        
        // Test that basic cache methods exist
        $this->assertTrue(method_exists($service, 'get'));
        $this->assertTrue(method_exists($service, 'set'));
        $this->assertTrue(method_exists($service, 'invalidateEntity'));
        $this->assertTrue(method_exists($service, 'invalidateEntities'));
        $this->assertTrue(method_exists($service, 'invalidateCacheKey'));
        
        // Test that warmup methods exist
        $this->assertTrue(method_exists($service, 'warmupShipmentLine'));
        $this->assertTrue(method_exists($service, 'warmupCustomerOrderLine'));
        $this->assertTrue(method_exists($service, 'warmupCustomerOrder'));
        $this->assertTrue(method_exists($service, 'warmupShipment'));
        
        // Test that invalidation methods exist
        $this->assertTrue(method_exists($service, 'invalidateAndWarmupByPrice'));
        $this->assertTrue(method_exists($service, 'invalidateAndWarmupByColourway'));
        $this->assertTrue(method_exists($service, 'invalidateAndWarmupByStyleVersion'));
        
        // Test that utility methods exist
        $this->assertTrue(method_exists($service, 'getCacheStats'));
        $this->assertTrue(method_exists($service, 'cleanupStale'));
    }

    /**
     * Test that PriceResolutionService invalidation methods return integers
     */
    public function test_price_resolution_service_invalidation_returns_integers()
    {
        $service = app(PriceResolutionService::class);
        
        // Mock the database calls to avoid database issues
        $mockService = Mockery::mock(PriceResolutionService::class)->makePartial();
        
        // Test that invalidation methods return integers
        $mockService->shouldReceive('invalidateByColourway')
            ->with(1)
            ->andReturn(5);
        $this->assertIsInt($mockService->invalidateByColourway(1));
        
        $mockService->shouldReceive('invalidateByCustomer')
            ->with(1)
            ->andReturn(3);
        $this->assertIsInt($mockService->invalidateByCustomer(1));
        
        $mockService->shouldReceive('invalidateBySupplier')
            ->with(1)
            ->andReturn(2);
        $this->assertIsInt($mockService->invalidateBySupplier(1));
        
        $mockService->shouldReceive('invalidateByPrice')
            ->with(1)
            ->andReturn(1);
        $this->assertIsInt($mockService->invalidateByPrice(1));
    }

    /**
     * Test that TotalCacheService cache operations work correctly
     */
    public function test_total_cache_service_operations()
    {
        $service = app(TotalCacheService::class);
        
        // Test set and get operations
        $testData = ['total' => 100.50, 'count' => 5];
        $service->set('test_entity', 1, 'test_key', $testData);
        
        $retrieved = $service->get('test_entity', 1, 'test_key');
        $this->assertEquals($testData, $retrieved);
        
        // Test invalidation
        $service->invalidateEntity('test_entity', 1);
        $retrievedAfterInvalidation = $service->get('test_entity', 1, 'test_key');
        $this->assertNull($retrievedAfterInvalidation);
    }

    /**
     * Test that cache invalidation methods handle exceptions gracefully
     */
    public function test_cache_invalidation_handles_exceptions()
    {
        $service = app(PriceResolutionService::class);
        
        // Test that methods don't throw exceptions with invalid IDs
        $this->assertIsInt($service->invalidateByColourway(999999));
        $this->assertIsInt($service->invalidateByCustomer(999999));
        $this->assertIsInt($service->invalidateBySupplier(999999));
        $this->assertIsInt($service->invalidateByPrice(999999));
    }

    /**
     * Test that cache stats methods return expected structure
     */
    public function test_cache_stats_structure()
    {
        $priceService = app(PriceResolutionService::class);
        $totalService = app(TotalCacheService::class);
        
        $priceStats = $priceService->getCacheStats();
        $totalStats = $totalService->getCacheStats();
        
        // Test price resolution stats structure
        $this->assertArrayHasKey('total', $priceStats);
        $this->assertArrayHasKey('fresh', $priceStats);
        $this->assertArrayHasKey('stale', $priceStats);
        $this->assertArrayHasKey('fresh_percentage', $priceStats);
        
        // Test total cache stats structure
        $this->assertArrayHasKey('total', $totalStats);
        $this->assertArrayHasKey('fresh', $totalStats);
        $this->assertArrayHasKey('stale', $totalStats);
        $this->assertArrayHasKey('by_entity', $totalStats);
        
        // Test that values are integers
        $this->assertIsInt($priceStats['total']);
        $this->assertIsInt($priceStats['fresh']);
        $this->assertIsInt($priceStats['stale']);
        $this->assertIsFloat($priceStats['fresh_percentage']);
        
        $this->assertIsInt($totalStats['total']);
        $this->assertIsInt($totalStats['fresh']);
        $this->assertIsInt($totalStats['stale']);
        $this->assertIsArray($totalStats['by_entity']);
    }

    /**
     * Test that cleanup methods return integers
     */
    public function test_cleanup_methods_return_integers()
    {
        $priceService = app(PriceResolutionService::class);
        $totalService = app(TotalCacheService::class);
        
        $priceCleanup = $priceService->cleanupStaleResolutions();
        $totalCleanup = $totalService->cleanupStale();
        
        $this->assertIsInt($priceCleanup);
        $this->assertIsInt($totalCleanup);
    }

    protected function tearDown(): void
    {
        Mockery::close();
        parent::tearDown();
    }
}
