<?php

namespace App\Imports;

use App\Models\Statement;
use App\Models\StatementTransaction;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Illuminate\Support\Collection;

class StatementImport implements ToCollection, WithHeadingRow
{
    public function __construct(
        protected ?string $overrideCardholder = null,
        protected ?string $overrideLast4 = null,
        protected ?string $overrideCycle = null,
    ) {}

    public function collection(Collection $rows)
    {
        if ($rows->isEmpty()) {
            return;
        }

        // Determine statement bounds from data
        $dates = $rows->pluck('transaction_date')->filter()->map(fn ($d) => Carbon::parse($this->excelDate($d)))->sort();
        $start = optional($dates->first())->toDateString();
        $end = optional($dates->last())->toDateString();

        $first = $rows->first();
        $cardholder = $this->overrideCardholder ?? (string)($first['cardholder_name'] ?? '');
        $last4 = $this->overrideLast4 ?? Str::of((string)($first['account_number'] ?? ''))
            ->replace(['*', ' '], '')->substr(-4);
        $cycle = $this->overrideCycle ?? (string)($first['statement_cycle'] ?? '');

        $statement = Statement::create([
            'cardholder_name' => $cardholder,
            'account_number_last4' => $last4,
            'start_date' => $start ?? now()->toDateString(),
            'end_date' => $end ?? now()->toDateString(),
            'statement_cycle' => $cycle,
            'status' => 'open',
        ]);

        foreach ($rows as $row) {
            StatementTransaction::create([
                'statement_id' => $statement->id,
                'cardholder_name' => (string)($row['cardholder_name'] ?? $cardholder),
                'account_number_last4' => $last4,
                'transaction_date' => Carbon::parse($this->excelDate($row['transaction_date'] ?? $row['posted_date'] ?? now()))->toDateString(),
                'merchant_name' => (string)($row['merchant_name'] ?? ''),
                'amount' => (float)($row['amount'] ?? 0),
                'currency' => (string)($row['currency'] ?? 'GBP'),
                'original_amount' => $row['original_amount'] ?? null,
                'original_currency' => $row['original_currency'] ?? null,
                'conversion_rate' => $row['conversion_rate'] ?? null,
                'posted_date' => isset($row['posted_date']) ? Carbon::parse($this->excelDate($row['posted_date']))->toDateString() : null,
                'transaction_time' => $row['transaction_time'] ?? null,
                'authorisation_code' => $row['authorisation_code'] ?? null,
                'transaction_id_ref' => $row['transaction_id'] ?? null,
                'merchant_category' => $row['merchant_category'] ?? null,
                'transaction_type' => $row['transaction_type'] ?? null,
                'mcc_description' => $row['mcc_description'] ?? null,
                'merchant_city' => $row['merchant_town/city'] ?? $row['merchant_city'] ?? null,
                'merchant_state' => $row['merchant_county/state'] ?? $row['merchant_state'] ?? null,
                'merchant_postcode' => $row['merchant_post_code/zipcode'] ?? $row['merchant_postcode'] ?? null,
                'mcc' => $row['mcc'] ?? null,
                'receipt_flag' => ($row['receipt?'] ?? false) ? true : false,
                'details' => $row['details_(travel_|_subsistence_|_entertainment_etc)'] ?? $row['details'] ?? null,
            ]);
        }
    }

    private function excelDate($value)
    {
        if (is_numeric($value)) {
            return Carbon::createFromTimestampUTC(((int)$value - 25569) * 86400);
        }
        return $value;
    }
}


