<?php

namespace App\Http\Controllers;

use App\Jobs\GenerateCustomSessionJob;
use App\Models\CustomRequest;
use App\Models\CustomSession;
use App\Services\Contracts\SafetyService;
use Illuminate\Http\Request;

class CustomController extends Controller
{
    public function store(Request $request, SafetyService $safety)
    {
        $data = $request->validate([
            'overrides' => ['nullable', 'array'],
            'overrides.topic' => ['nullable', 'string', 'max:64'],
            'overrides.length_min' => ['nullable', 'integer', 'min:5', 'max:60'],
            'overrides.voice' => ['nullable', 'string', 'max:64'],
        ]);

        $overrides = $data['overrides'] ?? [];
        $decision = $safety->check(json_encode($overrides), 'input');
        if ($decision->action === 'block') {
            return response()->json([
                'status' => 'blocked',
                'message' => 'Request blocked by safety policy',
            ], 422);
        }

        $req = CustomRequest::create([
            'user_id' => $request->user()->id,
            'source' => 'manual',
            'overrides' => $overrides,
            'status' => CustomRequest::STATUS_PENDING,
        ]);

        GenerateCustomSessionJob::dispatch($req)->onQueue('ai');

        return response()->json([
            'id' => $req->id,
            'status' => $req->status,
            'scheduled_for' => optional($req->scheduled_for)->toISOString(),
        ], 201);
    }

    public function showRequest(int $id)
    {
        $req = CustomRequest::findOrFail($id);
        $this->authorize('view', $req);

        return response()->json($req);
    }

    public function sessions(Request $request)
    {
        $sessions = CustomSession::whereIn('request_id', CustomRequest::where('user_id', $request->user()->id)->pluck('id'))
            ->latest()
            ->paginate(15);

        return response()->json($sessions);
    }

    public function showSession(int $id)
    {
        $session = CustomSession::findOrFail($id);
        $this->authorize('view', $session);

        return response()->json($session);
    }
}
