<?php

namespace App\Console\Commands;

use App\Models\Receipt;
use App\Services\ReceiptMergingService;
use Illuminate\Console\Command;

class TestReceiptMerging extends Command
{
    protected $signature = 'test:receipt-merging {receipt_ids* : Receipt IDs to merge}';
    protected $description = 'Test receipt merging functionality';

    public function handle(ReceiptMergingService $mergingService)
    {
        $receiptIds = $this->argument('receipt_ids');
        
        if (count($receiptIds) < 2) {
            $this->error('Please provide at least 2 receipt IDs to merge.');
            return;
        }

        $this->info('Testing Receipt Merging...');
        
        // Show original receipts
        $receipts = Receipt::whereIn('id', $receiptIds)->get();
        
        if ($receipts->count() !== count($receiptIds)) {
            $this->error('Some receipts not found.');
            return;
        }

        $this->info("Found {$receipts->count()} receipts to merge:");
        foreach ($receipts as $receipt) {
            $this->line("  ID: {$receipt->id} | {$receipt->merchant_name} | £{$receipt->total_amount} | {$receipt->receipt_date}");
        }

        try {
            // Dispatch the merge job
            \App\Jobs\MergeReceiptsJob::dispatch($receiptIds, $receipts->first()->user_id);
            
            $this->info("✅ Receipt merge job dispatched successfully!");
            $this->line("The merge will be processed in the background.");
            $this->line("Check the queue worker or logs to see the progress.");
            $this->line("The merged receipt will appear in the unmatched receipts list once complete.");
            
        } catch (\Exception $e) {
            $this->error("❌ Failed to dispatch merge job: " . $e->getMessage());
        }
    }
}
