<?php

namespace App\Console\Commands;

use App\Models\Receipt;
use App\Services\ReceiptMergingService;
use Illuminate\Console\Command;

class TestReceiptMerging extends Command
{
    protected $signature = 'test:receipt-merging {receipt_ids* : Receipt IDs to merge}';
    protected $description = 'Test receipt merging functionality';

    public function handle(ReceiptMergingService $mergingService)
    {
        $receiptIds = $this->argument('receipt_ids');
        
        if (count($receiptIds) < 2) {
            $this->error('Please provide at least 2 receipt IDs to merge.');
            return;
        }

        $this->info('Testing Receipt Merging...');
        
        // Show original receipts
        $receipts = Receipt::whereIn('id', $receiptIds)->get();
        
        if ($receipts->count() !== count($receiptIds)) {
            $this->error('Some receipts not found.');
            return;
        }

        $this->info("Found {$receipts->count()} receipts to merge:");
        foreach ($receipts as $receipt) {
            $this->line("  ID: {$receipt->id} | {$receipt->merchant_name} | £{$receipt->total_amount} | {$receipt->receipt_date}");
        }

        try {
            $mergedReceipt = $mergingService->mergeReceipts($receiptIds);
            
            $this->info("✅ Successfully merged receipts!");
            $this->line("Merged Receipt ID: {$mergedReceipt->id}");
            $this->line("Merchant: {$mergedReceipt->merchant_name}");
            $this->line("Total Amount: £{$mergedReceipt->total_amount}");
            $this->line("Date: {$mergedReceipt->receipt_date}");
            
            // Show merge info
            $mergeInfo = $mergingService->getMergedReceiptInfo($mergedReceipt);
            if ($mergeInfo) {
                $this->line("Merged from: " . implode(', ', $mergeInfo['merged_from']));
            }
            
        } catch (\Exception $e) {
            $this->error("❌ Failed to merge receipts: " . $e->getMessage());
        }
    }
}
