const db = require('./db-multi-tenant');

/**
 * Middleware to require superuser authentication
 */
function requireSuperuser(req, res, next) {
  if (!req.user) {
    return res.redirect('/login?returnTo=' + encodeURIComponent(req.originalUrl));
  }
  
  if (!req.user.is_superuser) {
    return res.status(403).send(`
      <!DOCTYPE html>
      <html>
      <head>
        <title>Access Denied</title>
        <style>
          body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            display: flex;
            align-items: center;
            justify-content: center;
            min-height: 100vh;
            margin: 0;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
          }
          .container {
            text-align: center;
            padding: 40px;
            background: rgba(255,255,255,0.1);
            border-radius: 16px;
            backdrop-filter: blur(10px);
          }
          h1 { font-size: 48px; margin: 0 0 16px 0; }
          p { font-size: 18px; margin: 0 0 24px 0; opacity: 0.9; }
          a {
            display: inline-block;
            padding: 12px 24px;
            background: white;
            color: #667eea;
            text-decoration: none;
            border-radius: 8px;
            font-weight: 600;
          }
        </style>
      </head>
      <body>
        <div class="container">
          <h1>🚫 Access Denied</h1>
          <p>You do not have permission to access the superuser portal.</p>
          <p>Only superuser accounts can access this area.</p>
          <a href="/dashboard">← Back to Dashboard</a>
        </div>
      </body>
      </html>
    `);
  }
  
  next();
}

/**
 * Middleware to ensure superuser portal only works on app subdomain
 */
function requireSuperuserSubdomain(req, res, next) {
  if (!req.tenantInfo.isAppSubdomain) {
    return res.status(404).send('Superuser portal is only available at app subdomain');
  }
  next();
}

/**
 * Log superuser action to audit log
 */
async function logAdminAction(adminUserId, actionType, targetUserId = null, actionDetails = {}, req = null) {
  try {
    const ipAddress = req ? (req.ip || req.connection.remoteAddress) : null;
    const userAgent = req ? req.headers['user-agent'] : null;
    
    await db.pool.query(
      `INSERT INTO admin_audit_log (admin_user_id, action_type, target_user_id, action_details, ip_address, user_agent)
       VALUES ($1, $2, $3, $4, $5, $6)`,
      [adminUserId, actionType, targetUserId, JSON.stringify(actionDetails), ipAddress, userAgent]
    );
  } catch (error) {
    console.error('Failed to log admin action:', error);
    // Don't throw - logging should not break the main action
  }
}

/**
 * Get all users with pagination and filtering
 */
async function getAllUsers(options = {}) {
  const { 
    page = 1, 
    limit = 50, 
    search = null, 
    tier = null, 
    status = null,
    sortBy = 'created_at',
    sortOrder = 'DESC'
  } = options;
  
  const offset = (page - 1) * limit;
  const allowedSortFields = ['created_at', 'username', 'email', 'subscription_tier', 'subscription_status', 'current_photo_count', 'last_login_at'];
  const orderField = allowedSortFields.includes(sortBy) ? sortBy : 'created_at';
  const orderDir = sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
  
  let whereConditions = [];
  let queryParams = [];
  let paramIndex = 1;
  
  if (search) {
    whereConditions.push(`(username ILIKE $${paramIndex} OR email ILIKE $${paramIndex} OR full_name ILIKE $${paramIndex})`);
    queryParams.push(`%${search}%`);
    paramIndex++;
  }
  
  if (tier) {
    whereConditions.push(`subscription_tier = $${paramIndex}`);
    queryParams.push(tier);
    paramIndex++;
  }
  
  if (status) {
    whereConditions.push(`subscription_status = $${paramIndex}`);
    queryParams.push(status);
    paramIndex++;
  }
  
  const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';
  
  // Get total count
  const countResult = await db.pool.query(
    `SELECT COUNT(*) as total FROM users ${whereClause}`,
    queryParams
  );
  const total = parseInt(countResult.rows[0].total, 10);
  
  // Get users
  const usersResult = await db.pool.query(
    `SELECT id, username, email, full_name, subscription_tier, subscription_status, 
            is_superuser, is_suspended, trial_ends_at, subscription_ends_at,
            current_photo_count, max_photos, current_storage_bytes, max_storage_bytes,
            custom_domain, custom_domain_verified, stripe_customer_id,
            created_at, last_login_at
     FROM users
     ${whereClause}
     ORDER BY ${orderField} ${orderDir}
     LIMIT $${paramIndex} OFFSET $${paramIndex + 1}`,
    [...queryParams, limit, offset]
  );
  
  return {
    users: usersResult.rows,
    pagination: {
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit),
      hasNext: page < Math.ceil(total / limit),
      hasPrev: page > 1
    }
  };
}

/**
 * Get user statistics for dashboard
 */
async function getUserStatistics() {
  const result = await db.pool.query(`
    SELECT 
      COALESCE(COUNT(*), 0) as total_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_status = 'trialing'), 0) as trialing_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_status = 'active'), 0) as active_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_status = 'past_due'), 0) as past_due_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_status = 'canceled'), 0) as canceled_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_status = 'suspended'), 0) as suspended_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_tier = 'starter'), 0) as starter_tier_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_tier = 'professional'), 0) as professional_tier_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_tier = 'business'), 0) as business_tier_users,
      COALESCE(COUNT(*) FILTER (WHERE created_at > NOW() - INTERVAL '30 days'), 0) as new_users_30d,
      COALESCE(COUNT(*) FILTER (WHERE last_login_at > NOW() - INTERVAL '7 days'), 0) as active_last_7d,
      COALESCE(SUM(current_photo_count), 0) as total_photos,
      COALESCE(SUM(current_storage_bytes), 0) as total_storage_bytes
    FROM users
    WHERE is_superuser = FALSE
  `);
  
  return result.rows[0];
}

/**
 * Get platform revenue statistics
 */
async function getRevenueStatistics() {
  const tiersResult = await db.pool.query(`
    SELECT st.tier_name, st.price_pence, COUNT(u.id) as user_count
    FROM subscription_tiers st
    LEFT JOIN users u ON u.subscription_tier = st.tier_key 
      AND u.subscription_status IN ('active', 'trialing')
      AND u.is_superuser = FALSE
    WHERE st.is_active = TRUE
    GROUP BY st.tier_name, st.price_pence
    ORDER BY st.display_order
  `);
  
  let mrr = 0; // Monthly Recurring Revenue in pence
  const breakdown = [];
  
  for (const tier of tiersResult.rows) {
    const tierMrr = tier.price_pence * tier.user_count;
    mrr += tierMrr;
    breakdown.push({
      tier: tier.tier_name,
      users: tier.user_count,
      price: tier.price_pence,
      revenue: tierMrr
    });
  }
  
  return {
    mrr, // in pence
    mrrFormatted: `£${(mrr / 100).toFixed(2)}`,
    arr: mrr * 12, // Annual Recurring Revenue
    arrFormatted: `£${((mrr * 12) / 100).toFixed(2)}`,
    breakdown
  };
}

/**
 * Update user subscription tier
 */
async function updateUserTier(adminUserId, targetUserId, newTier, req = null) {
  const user = await db.getUserById(targetUserId);
  if (!user) throw new Error('User not found');
  
  const oldTier = user.subscription_tier;
  
  await db.updateSubscriptionTier(targetUserId, newTier);
  
  await logAdminAction(adminUserId, 'tier_assigned', targetUserId, {
    old_tier: oldTier,
    new_tier: newTier
  }, req);
  
  return true;
}

/**
 * Suspend user account
 */
async function suspendUser(adminUserId, targetUserId, reason = null, req = null) {
  await db.pool.query(
    `UPDATE users 
     SET is_suspended = TRUE, 
         subscription_status = 'suspended',
         updated_at = CURRENT_TIMESTAMP
     WHERE id = $1`,
    [targetUserId]
  );
  
  await logAdminAction(adminUserId, 'user_suspended', targetUserId, { reason }, req);
  
  return true;
}

/**
 * Activate suspended user account
 */
async function activateUser(adminUserId, targetUserId, req = null) {
  const user = await db.getUserById(targetUserId);
  if (!user) throw new Error('User not found');
  
  // Restore to active status if they had a subscription, otherwise trialing
  const newStatus = user.stripe_subscription_id ? 'active' : 'trialing';
  
  await db.pool.query(
    `UPDATE users 
     SET is_suspended = FALSE, 
         subscription_status = $1,
         updated_at = CURRENT_TIMESTAMP
     WHERE id = $2`,
    [newStatus, targetUserId]
  );
  
  await logAdminAction(adminUserId, 'user_activated', targetUserId, { new_status: newStatus }, req);
  
  return true;
}

/**
 * Update user limits manually
 */
async function updateUserLimits(adminUserId, targetUserId, maxPhotos, maxStorageBytes, req = null) {
  await db.pool.query(
    `UPDATE users 
     SET max_photos = $1, 
         max_storage_bytes = $2,
         updated_at = CURRENT_TIMESTAMP
     WHERE id = $3`,
    [maxPhotos, maxStorageBytes, targetUserId]
  );
  
  await logAdminAction(adminUserId, 'limits_adjusted', targetUserId, {
    max_photos: maxPhotos,
    max_storage_bytes: maxStorageBytes
  }, req);
  
  return true;
}

/**
 * Delete user account permanently
 */
async function deleteUser(adminUserId, targetUserId, req = null) {
  const user = await db.getUserById(targetUserId);
  if (!user) throw new Error('User not found');
  
  // Don't allow deleting other superusers
  if (user.is_superuser) {
    throw new Error('Cannot delete superuser accounts');
  }
  
  await logAdminAction(adminUserId, 'user_deleted', targetUserId, {
    username: user.username,
    email: user.email,
    subscription_tier: user.subscription_tier
  }, req);
  
  await db.pool.query('DELETE FROM users WHERE id = $1', [targetUserId]);
  
  return true;
}

/**
 * Get all subscription tiers
 */
async function getAllTiers() {
  const result = await db.pool.query(
    `SELECT * FROM subscription_tiers ORDER BY display_order ASC, tier_name ASC`
  );
  return result.rows;
}

/**
 * Get tier by ID
 */
async function getTierById(tierId) {
  const result = await db.pool.query(
    'SELECT * FROM subscription_tiers WHERE id = $1',
    [tierId]
  );
  return result.rows[0] || null;
}

/**
 * Create new subscription tier
 */
async function createTier(adminUserId, tierData, req = null) {
  const {
    tier_key,
    tier_name,
    price_pence,
    max_photos,
    max_storage_bytes,
    allows_custom_domain = false,
    allows_multiple_domains = false,
    description = null,
    features = []
  } = tierData;
  
  const result = await db.pool.query(
    `INSERT INTO subscription_tiers 
     (tier_key, tier_name, price_pence, max_photos, max_storage_bytes, 
      allows_custom_domain, allows_multiple_domains, description, features)
     VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
     RETURNING *`,
    [tier_key, tier_name, price_pence, max_photos, max_storage_bytes,
     allows_custom_domain, allows_multiple_domains, description, JSON.stringify(features)]
  );
  
  await logAdminAction(adminUserId, 'tier_created', null, {
    tier_id: result.rows[0].id,
    tier_key,
    tier_name
  }, req);
  
  return result.rows[0];
}

/**
 * Update subscription tier
 */
async function updateTier(adminUserId, tierId, tierData, req = null) {
  const fields = [];
  const values = [];
  let paramIndex = 1;
  
  const allowedFields = [
    'tier_name', 'price_pence', 'max_photos', 'max_storage_bytes',
    'allows_custom_domain', 'allows_multiple_domains', 'display_order',
    'is_active', 'description', 'features'
  ];
  
  for (const [key, value] of Object.entries(tierData)) {
    if (allowedFields.includes(key) && value !== undefined) {
      fields.push(`${key} = $${paramIndex}`);
      values.push(key === 'features' ? JSON.stringify(value) : value);
      paramIndex++;
    }
  }
  
  if (fields.length === 0) return null;
  
  values.push(tierId);
  const result = await db.pool.query(
    `UPDATE subscription_tiers 
     SET ${fields.join(', ')}, updated_at = CURRENT_TIMESTAMP
     WHERE id = $${paramIndex}
     RETURNING *`,
    values
  );
  
  await logAdminAction(adminUserId, 'tier_updated', null, {
    tier_id: tierId,
    updated_fields: Object.keys(tierData)
  }, req);
  
  return result.rows[0];
}

/**
 * Deactivate (soft delete) subscription tier
 */
async function deactivateTier(adminUserId, tierId, req = null) {
  await db.pool.query(
    'UPDATE subscription_tiers SET is_active = FALSE, updated_at = CURRENT_TIMESTAMP WHERE id = $1',
    [tierId]
  );
  
  await logAdminAction(adminUserId, 'tier_deleted', null, { tier_id: tierId }, req);
  
  return true;
}

/**
 * Get recent admin actions
 */
async function getRecentAdminActions(limit = 50) {
  const result = await db.pool.query(
    `SELECT aal.*, 
            u.username as admin_username,
            u.email as admin_email,
            tu.username as target_username,
            tu.email as target_email
     FROM admin_audit_log aal
     JOIN users u ON aal.admin_user_id = u.id
     LEFT JOIN users tu ON aal.target_user_id = tu.id
     ORDER BY aal.created_at DESC
     LIMIT $1`,
    [limit]
  );
  return result.rows;
}

/**
 * Make a user a superuser
 */
async function makeSuperuser(adminUserId, targetUserId, req = null) {
  await db.pool.query(
    'UPDATE users SET is_superuser = TRUE, updated_at = CURRENT_TIMESTAMP WHERE id = $1',
    [targetUserId]
  );
  
  await logAdminAction(adminUserId, 'user_updated', targetUserId, {
    change: 'promoted_to_superuser'
  }, req);
  
  return true;
}

/**
 * Remove superuser status
 */
async function removeSuperuser(adminUserId, targetUserId, req = null) {
  // Don't allow removing own superuser status
  if (adminUserId === targetUserId) {
    throw new Error('Cannot remove your own superuser status');
  }
  
  await db.pool.query(
    'UPDATE users SET is_superuser = FALSE, updated_at = CURRENT_TIMESTAMP WHERE id = $1',
    [targetUserId]
  );
  
  await logAdminAction(adminUserId, 'user_updated', targetUserId, {
    change: 'removed_superuser_status'
  }, req);
  
  return true;
}

module.exports = {
  requireSuperuser,
  requireSuperuserSubdomain,
  logAdminAction,
  getAllUsers,
  getUserStatistics,
  getRevenueStatistics,
  updateUserTier,
  suspendUser,
  activateUser,
  updateUserLimits,
  deleteUser,
  getAllTiers,
  getTierById,
  createTier,
  updateTier,
  deactivateTier,
  getRecentAdminActions,
  makeSuperuser,
  removeSuperuser
};

