const db = require('./db-multi-tenant');

/**
 * Middleware to require superuser authentication
 */
function requireSuperuser(req, res, next) {
  if (!req.user) {
    return res.redirect('/login?returnTo=' + encodeURIComponent(req.originalUrl));
  }
  
  if (!req.user.is_superuser) {
    return res.status(403).send(`
      <!DOCTYPE html>
      <html>
      <head>
        <title>Access Denied</title>
        <style>
          body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            display: flex;
            align-items: center;
            justify-content: center;
            min-height: 100vh;
            margin: 0;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
          }
          .container {
            text-align: center;
            padding: 40px;
            background: rgba(255,255,255,0.1);
            border-radius: 16px;
            backdrop-filter: blur(10px);
          }
          h1 { font-size: 48px; margin: 0 0 16px 0; }
          p { font-size: 18px; margin: 0 0 24px 0; opacity: 0.9; }
          a {
            display: inline-block;
            padding: 12px 24px;
            background: white;
            color: #667eea;
            text-decoration: none;
            border-radius: 8px;
            font-weight: 600;
          }
        </style>
      </head>
      <body>
        <div class="container">
          <h1>🚫 Access Denied</h1>
          <p>You do not have permission to access the superuser portal.</p>
          <p>Only superuser accounts can access this area.</p>
          <a href="/dashboard">← Back to Dashboard</a>
        </div>
      </body>
      </html>
    `);
  }
  
  next();
}

/**
 * Middleware to ensure superuser portal only works on app subdomain
 */
function requireSuperuserSubdomain(req, res, next) {
  if (!req.tenantInfo.isAppSubdomain) {
    return res.status(404).send('Superuser portal is only available at app subdomain');
  }
  next();
}

/**
 * Log superuser action to audit log
 */
async function logAdminAction(adminUserId, actionType, targetUserId = null, actionDetails = {}, req = null) {
  try {
    const ipAddress = req ? (req.ip || req.connection.remoteAddress) : null;
    const userAgent = req ? req.headers['user-agent'] : null;
    
    await db.pool.query(
      `INSERT INTO admin_audit_log (admin_user_id, action_type, target_user_id, action_details, ip_address, user_agent)
       VALUES ($1, $2, $3, $4, $5, $6)`,
      [adminUserId, actionType, targetUserId, JSON.stringify(actionDetails), ipAddress, userAgent]
    );
  } catch (error) {
    console.error('Failed to log admin action:', error);
    // Don't throw - logging should not break the main action
  }
}

/**
 * Get all users with pagination and filtering
 */
async function getAllUsers(options = {}) {
  const { 
    page = 1, 
    limit = 50, 
    search = null, 
    tier = null, 
    status = null,
    sortBy = 'created_at',
    sortOrder = 'DESC'
  } = options;
  
  const offset = (page - 1) * limit;
  const allowedSortFields = ['created_at', 'username', 'email', 'subscription_tier', 'subscription_status', 'current_photo_count', 'last_login_at'];
  const orderField = allowedSortFields.includes(sortBy) ? sortBy : 'created_at';
  const orderDir = sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
  
  let whereConditions = [];
  let queryParams = [];
  let paramIndex = 1;
  
  if (search) {
    whereConditions.push(`(username ILIKE $${paramIndex} OR email ILIKE $${paramIndex} OR full_name ILIKE $${paramIndex})`);
    queryParams.push(`%${search}%`);
    paramIndex++;
  }
  
  if (tier) {
    whereConditions.push(`subscription_tier = $${paramIndex}`);
    queryParams.push(tier);
    paramIndex++;
  }
  
  if (status) {
    whereConditions.push(`subscription_status = $${paramIndex}`);
    queryParams.push(status);
    paramIndex++;
  }
  
  const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';
  
  // Get total count
  const countResult = await db.pool.query(
    `SELECT COUNT(*) as total FROM users ${whereClause}`,
    queryParams
  );
  const total = parseInt(countResult.rows[0].total, 10);
  
  // Get users
  const usersResult = await db.pool.query(
    `SELECT id, username, email, full_name, subscription_tier, subscription_status, 
            is_superuser, is_suspended, trial_ends_at, subscription_ends_at,
            current_photo_count, max_photos, current_storage_bytes, max_storage_bytes,
            custom_domain, custom_domain_verified, stripe_customer_id, stripe_subscription_id,
            skip_payment_enforcement, last_payment_check_at, payment_grace_period_ends,
            created_at, last_login_at
     FROM users
     ${whereClause}
     ORDER BY ${orderField} ${orderDir}
     LIMIT $${paramIndex} OFFSET $${paramIndex + 1}`,
    [...queryParams, limit, offset]
  );
  
  return {
    users: usersResult.rows,
    pagination: {
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit),
      hasNext: page < Math.ceil(total / limit),
      hasPrev: page > 1
    }
  };
}

/**
 * Get user statistics for dashboard
 */
async function getUserStatistics() {
  const result = await db.pool.query(`
    SELECT 
      COALESCE(COUNT(*), 0) as total_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_status = 'trialing'), 0) as trialing_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_status = 'active'), 0) as active_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_status = 'past_due'), 0) as past_due_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_status = 'canceled'), 0) as canceled_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_status = 'suspended'), 0) as suspended_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_tier = 'starter'), 0) as starter_tier_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_tier = 'professional'), 0) as professional_tier_users,
      COALESCE(COUNT(*) FILTER (WHERE subscription_tier = 'business'), 0) as business_tier_users,
      COALESCE(COUNT(*) FILTER (WHERE created_at > NOW() - INTERVAL '30 days'), 0) as new_users_30d,
      COALESCE(COUNT(*) FILTER (WHERE last_login_at > NOW() - INTERVAL '7 days'), 0) as active_last_7d,
      COALESCE(SUM(current_photo_count), 0) as total_photos,
      COALESCE(SUM(current_storage_bytes), 0) as total_storage_bytes
    FROM users
    WHERE is_superuser = FALSE
  `);
  
  return result.rows[0];
}

/**
 * Get platform revenue statistics
 */
async function getRevenueStatistics() {
  const tiersResult = await db.pool.query(`
    SELECT st.tier_name, st.price_pence, st.display_order, COUNT(u.id) as user_count
    FROM subscription_tiers st
    LEFT JOIN users u ON u.subscription_tier = st.tier_key 
      AND u.subscription_status IN ('active', 'trialing')
      AND u.is_superuser = FALSE
    WHERE st.is_active = TRUE
    GROUP BY st.tier_name, st.price_pence, st.display_order
    ORDER BY st.display_order
  `);
  
  let mrr = 0; // Monthly Recurring Revenue in pence
  const breakdown = [];
  
  for (const tier of tiersResult.rows) {
    const tierMrr = tier.price_pence * tier.user_count;
    mrr += tierMrr;
    breakdown.push({
      tier: tier.tier_name,
      users: tier.user_count,
      price: tier.price_pence,
      revenue: tierMrr
    });
  }
  
  return {
    mrr, // in pence
    mrrFormatted: `£${(mrr / 100).toFixed(2)}`,
    arr: mrr * 12, // Annual Recurring Revenue
    arrFormatted: `£${((mrr * 12) / 100).toFixed(2)}`,
    breakdown
  };
}

/**
 * Update user subscription tier
 */
async function updateUserTier(adminUserId, targetUserId, newTier, req = null) {
  const user = await db.getUserById(targetUserId);
  if (!user) throw new Error('User not found');
  
  const oldTier = user.subscription_tier;
  
  await db.updateSubscriptionTier(targetUserId, newTier);
  
  await logAdminAction(adminUserId, 'tier_assigned', targetUserId, {
    old_tier: oldTier,
    new_tier: newTier
  }, req);
  
  return true;
}

/**
 * Suspend user account
 */
async function suspendUser(adminUserId, targetUserId, reason = null, req = null) {
  await db.pool.query(
    `UPDATE users 
     SET is_suspended = TRUE, 
         subscription_status = 'suspended',
         updated_at = CURRENT_TIMESTAMP
     WHERE id = $1`,
    [targetUserId]
  );
  
  await logAdminAction(adminUserId, 'user_suspended', targetUserId, { reason }, req);
  
  return true;
}

/**
 * Activate suspended user account
 */
async function activateUser(adminUserId, targetUserId, req = null) {
  const user = await db.getUserById(targetUserId);
  if (!user) throw new Error('User not found');
  
  // Restore to active status if they had a subscription, otherwise trialing
  const newStatus = user.stripe_subscription_id ? 'active' : 'trialing';
  
  await db.pool.query(
    `UPDATE users 
     SET is_suspended = FALSE, 
         subscription_status = $1,
         updated_at = CURRENT_TIMESTAMP
     WHERE id = $2`,
    [newStatus, targetUserId]
  );
  
  await logAdminAction(adminUserId, 'user_activated', targetUserId, { new_status: newStatus }, req);
  
  return true;
}

/**
 * Update user limits manually
 */
async function updateUserLimits(adminUserId, targetUserId, maxPhotos, maxStorageBytes, req = null) {
  await db.pool.query(
    `UPDATE users 
     SET max_photos = $1, 
         max_storage_bytes = $2,
         updated_at = CURRENT_TIMESTAMP
     WHERE id = $3`,
    [maxPhotos, maxStorageBytes, targetUserId]
  );
  
  await logAdminAction(adminUserId, 'limits_adjusted', targetUserId, {
    max_photos: maxPhotos,
    max_storage_bytes: maxStorageBytes
  }, req);
  
  return true;
}

/**
 * Delete user account permanently
 */
async function deleteUser(adminUserId, targetUserId, req = null) {
  const user = await db.getUserById(targetUserId);
  if (!user) throw new Error('User not found');
  
  // Don't allow deleting other superusers
  if (user.is_superuser) {
    throw new Error('Cannot delete superuser accounts');
  }
  
  await logAdminAction(adminUserId, 'user_deleted', targetUserId, {
    username: user.username,
    email: user.email,
    subscription_tier: user.subscription_tier
  }, req);
  
  await db.pool.query('DELETE FROM users WHERE id = $1', [targetUserId]);
  
  return true;
}

/**
 * Get all subscription tiers
 */
async function getAllTiers() {
  const result = await db.pool.query(
    `SELECT * FROM subscription_tiers ORDER BY display_order ASC, tier_name ASC`
  );
  return result.rows;
}

/**
 * Get tier by ID
 */
async function getTierById(tierId) {
  const result = await db.pool.query(
    'SELECT * FROM subscription_tiers WHERE id = $1',
    [tierId]
  );
  return result.rows[0] || null;
}

/**
 * Create new subscription tier
 */
async function createTier(adminUserId, tierData, req = null) {
  const {
    tier_key,
    tier_name,
    price_pence,
    max_photos,
    max_storage_bytes,
    allows_custom_domain = false,
    allows_multiple_domains = false,
    description = null,
    features = [],
    stripe_price_id = null,
    stripe_product_id = null
  } = tierData;
  
  const result = await db.pool.query(
    `INSERT INTO subscription_tiers 
     (tier_key, tier_name, price_pence, max_photos, max_storage_bytes, 
      allows_custom_domain, allows_multiple_domains, description, features, 
      stripe_price_id, stripe_product_id)
     VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
     RETURNING *`,
    [tier_key, tier_name, price_pence, max_photos, max_storage_bytes,
     allows_custom_domain, allows_multiple_domains, description, JSON.stringify(features),
     stripe_price_id, stripe_product_id]
  );
  
  await logAdminAction(adminUserId, 'tier_created', null, {
    tier_id: result.rows[0].id,
    tier_key,
    tier_name
  }, req);
  
  return result.rows[0];
}

/**
 * Update subscription tier
 */
async function updateTier(adminUserId, tierId, tierData, req = null) {
  const fields = [];
  const values = [];
  let paramIndex = 1;
  
  const allowedFields = [
    'tier_name', 'price_pence', 'max_photos', 'max_storage_bytes',
    'allows_custom_domain', 'allows_multiple_domains', 'display_order',
    'is_active', 'description', 'features', 'stripe_price_id', 'stripe_product_id'
  ];
  
  for (const [key, value] of Object.entries(tierData)) {
    if (allowedFields.includes(key) && value !== undefined) {
      fields.push(`${key} = $${paramIndex}`);
      values.push(key === 'features' ? JSON.stringify(value) : value);
      paramIndex++;
    }
  }
  
  if (fields.length === 0) return null;
  
  values.push(tierId);
  const result = await db.pool.query(
    `UPDATE subscription_tiers 
     SET ${fields.join(', ')}, updated_at = CURRENT_TIMESTAMP
     WHERE id = $${paramIndex}
     RETURNING *`,
    values
  );
  
  await logAdminAction(adminUserId, 'tier_updated', null, {
    tier_id: tierId,
    updated_fields: Object.keys(tierData)
  }, req);
  
  return result.rows[0];
}

/**
 * Deactivate (soft delete) subscription tier
 */
async function deactivateTier(adminUserId, tierId, req = null) {
  await db.pool.query(
    'UPDATE subscription_tiers SET is_active = FALSE, updated_at = CURRENT_TIMESTAMP WHERE id = $1',
    [tierId]
  );
  
  await logAdminAction(adminUserId, 'tier_deleted', null, { tier_id: tierId }, req);
  
  return true;
}

/**
 * Get recent admin actions
 */
async function getRecentAdminActions(limit = 50) {
  const result = await db.pool.query(
    `SELECT aal.*, 
            u.username as admin_username,
            u.email as admin_email,
            tu.username as target_username,
            tu.email as target_email
     FROM admin_audit_log aal
     JOIN users u ON aal.admin_user_id = u.id
     LEFT JOIN users tu ON aal.target_user_id = tu.id
     ORDER BY aal.created_at DESC
     LIMIT $1`,
    [limit]
  );
  return result.rows;
}

/**
 * Make a user a superuser
 */
async function makeSuperuser(adminUserId, targetUserId, req = null) {
  await db.pool.query(
    'UPDATE users SET is_superuser = TRUE, updated_at = CURRENT_TIMESTAMP WHERE id = $1',
    [targetUserId]
  );
  
  await logAdminAction(adminUserId, 'user_updated', targetUserId, {
    change: 'promoted_to_superuser'
  }, req);
  
  return true;
}

/**
 * Remove superuser status
 */
async function removeSuperuser(adminUserId, targetUserId, req = null) {
  // Don't allow removing own superuser status
  if (adminUserId === targetUserId) {
    throw new Error('Cannot remove your own superuser status');
  }
  
  await db.pool.query(
    'UPDATE users SET is_superuser = FALSE, updated_at = CURRENT_TIMESTAMP WHERE id = $1',
    [targetUserId]
  );
  
  await logAdminAction(adminUserId, 'user_updated', targetUserId, {
    change: 'removed_superuser_status'
  }, req);
  
  return true;
}

/**
 * Toggle payment enforcement for a user
 */
async function togglePaymentEnforcement(adminUserId, targetUserId, skipEnforcement, req = null) {
  await db.pool.query(
    `UPDATE users 
     SET skip_payment_enforcement = $1, 
         updated_at = CURRENT_TIMESTAMP 
     WHERE id = $2`,
    [skipEnforcement, targetUserId]
  );
  
  await logAdminAction(adminUserId, 'user_updated', targetUserId, {
    change: 'payment_enforcement_toggled',
    skip_enforcement: skipEnforcement
  }, req);
  
  return true;
}

/**
 * Create a new user manually (bypassing normal registration)
 */
async function createUserManually(adminUserId, userData, req = null) {
  const {
    username,
    email,
    password,
    full_name = null,
    subscription_tier = 'starter',
    subscription_status = 'trialing'
  } = userData;
  
  // Validate username (alphanumeric and hyphens only)
  const usernameRegex = /^[a-z0-9-]+$/;
  if (!usernameRegex.test(username)) {
    throw new Error('Username can only contain lowercase letters, numbers, and hyphens');
  }
  
  // Check minimum length
  if (username.length < 3 || username.length > 50) {
    throw new Error('Username must be between 3 and 50 characters');
  }
  
  // Check if username is taken
  const existingUsername = await db.getUserByUsername(username);
  if (existingUsername) {
    throw new Error('Username is already taken');
  }
  
  // Check if email is taken
  const existingEmail = await db.getUserByEmail(email);
  if (existingEmail) {
    throw new Error('Email is already registered');
  }
  
  // Validate email format
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  if (!emailRegex.test(email)) {
    throw new Error('Invalid email format');
  }
  
  // Validate password strength
  if (password.length < 8) {
    throw new Error('Password must be at least 8 characters long');
  }
  
  // Create user
  const bcrypt = require('bcrypt');
  const passwordHash = await bcrypt.hash(password, 10);
  const trialEndsAt = new Date(Date.now() + 14 * 24 * 60 * 60 * 1000); // 14 days from now
  
  const result = await db.pool.query(
    `INSERT INTO users (username, email, password_hash, full_name, subscription_tier, subscription_status, trial_ends_at)
     VALUES ($1, $2, $3, $4, $5, $6, $7)
     RETURNING id, username, email, full_name, subscription_tier, subscription_status, trial_ends_at, created_at`,
    [username.toLowerCase().trim(), email.toLowerCase().trim(), passwordHash, full_name, subscription_tier, subscription_status, trialEndsAt]
  );
  
  const newUser = result.rows[0];
  
  await logAdminAction(adminUserId, 'user_created', newUser.id, {
    username: newUser.username,
    email: newUser.email,
    subscription_tier: subscription_tier,
    subscription_status: subscription_status
  }, req);
  
  return newUser;
}

// ============================================================================
// PROMOTIONAL OFFERS MANAGEMENT
// ============================================================================

/**
 * Get all promotional offers with optional filtering
 */
async function getAllOffers(options = {}) {
  const { 
    active_only = false,
    tier_id = null,
    include_expired = false
  } = options;
  
  let whereConditions = [];
  let queryParams = [];
  let paramIndex = 1;
  
  if (active_only) {
    whereConditions.push(`po.is_active = TRUE`);
  }
  
  if (tier_id) {
    whereConditions.push(`po.target_tier_id = $${paramIndex}`);
    queryParams.push(tier_id);
    paramIndex++;
  }
  
  if (!include_expired) {
    whereConditions.push(`(po.valid_until IS NULL OR po.valid_until > CURRENT_TIMESTAMP)`);
  }
  
  const whereClause = whereConditions.length > 0 ? 'WHERE ' + whereConditions.join(' AND ') : '';
  
  const result = await db.pool.query(
    `SELECT po.*, 
            st.tier_key, st.tier_name,
            u.username as created_by_username
     FROM promotional_offers po
     JOIN subscription_tiers st ON po.target_tier_id = st.id
     LEFT JOIN users u ON po.created_by = u.id
     ${whereClause}
     ORDER BY po.created_at DESC`,
    queryParams
  );
  
  return result.rows;
}

/**
 * Get offer by ID
 */
async function getOfferById(offerId) {
  const result = await db.pool.query(
    `SELECT po.*, 
            st.tier_key, st.tier_name,
            u.username as created_by_username
     FROM promotional_offers po
     JOIN subscription_tiers st ON po.target_tier_id = st.id
     LEFT JOIN users u ON po.created_by = u.id
     WHERE po.id = $1`,
    [offerId]
  );
  
  return result.rows[0] || null;
}

/**
 * Get offer by code
 */
async function getOfferByCode(offerCode) {
  const result = await db.pool.query(
    `SELECT po.*, 
            st.tier_key, st.tier_name
     FROM promotional_offers po
     JOIN subscription_tiers st ON po.target_tier_id = st.id
     WHERE po.offer_code = $1`,
    [offerCode.toUpperCase()]
  );
  
  return result.rows[0] || null;
}

/**
 * Create a new promotional offer
 */
async function createOffer(adminUserId, offerData, req = null) {
  const {
    offer_code,
    offer_name,
    description,
    offer_type,
    target_tier_id,
    trial_days_override,
    discount_type,
    discount_value,
    discount_duration_months,
    custom_max_photos,
    custom_max_storage_bytes,
    custom_price_pence,
    max_redemptions,
    max_redemptions_per_user = 1,
    valid_from,
    valid_until,
    is_active = true,
    is_public = false,
    internal_notes
  } = offerData;
  
  // Validate offer code format (uppercase, alphanumeric + hyphens)
  const codeUpper = offer_code.toUpperCase().replace(/\s+/g, '-');
  if (!/^[A-Z0-9-]+$/.test(codeUpper)) {
    throw new Error('Offer code must contain only letters, numbers, and hyphens');
  }
  
  const result = await db.pool.query(
    `INSERT INTO promotional_offers (
       offer_code, offer_name, description, offer_type, target_tier_id,
       trial_days_override, discount_type, discount_value, discount_duration_months,
       custom_max_photos, custom_max_storage_bytes, custom_price_pence,
       max_redemptions, max_redemptions_per_user,
       valid_from, valid_until, is_active, is_public, internal_notes, created_by
     ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18, $19, $20)
     RETURNING *`,
    [
      codeUpper, offer_name, description, offer_type, target_tier_id,
      trial_days_override, discount_type, discount_value, discount_duration_months,
      custom_max_photos, custom_max_storage_bytes, custom_price_pence,
      max_redemptions, max_redemptions_per_user,
      valid_from || null, valid_until || null, is_active, is_public, internal_notes, adminUserId
    ]
  );
  
  await logAdminAction(adminUserId, 'offer_created', null, {
    offer_id: result.rows[0].id,
    offer_code: codeUpper,
    offer_type
  }, req);
  
  return result.rows[0];
}

/**
 * Update promotional offer
 */
async function updateOffer(adminUserId, offerId, offerData, req = null) {
  const fields = [];
  const values = [];
  let paramIndex = 1;
  
  const allowedFields = [
    'offer_name', 'description', 'trial_days_override',
    'discount_type', 'discount_value', 'discount_duration_months',
    'custom_max_photos', 'custom_max_storage_bytes', 'custom_price_pence',
    'max_redemptions', 'max_redemptions_per_user',
    'valid_from', 'valid_until', 'is_active', 'is_public', 'internal_notes'
  ];
  
  for (const [key, value] of Object.entries(offerData)) {
    if (allowedFields.includes(key) && value !== undefined) {
      fields.push(`${key} = $${paramIndex}`);
      values.push(value);
      paramIndex++;
    }
  }
  
  if (fields.length === 0) {
    throw new Error('No valid fields to update');
  }
  
  values.push(offerId);
  
  const result = await db.pool.query(
    `UPDATE promotional_offers 
     SET ${fields.join(', ')}
     WHERE id = $${paramIndex}
     RETURNING *`,
    values
  );
  
  if (result.rows.length === 0) {
    throw new Error('Offer not found');
  }
  
  await logAdminAction(adminUserId, 'offer_updated', null, {
    offer_id: offerId,
    changes: offerData
  }, req);
  
  return result.rows[0];
}

/**
 * Deactivate offer
 */
async function deactivateOffer(adminUserId, offerId, req = null) {
  const result = await db.pool.query(
    'UPDATE promotional_offers SET is_active = FALSE WHERE id = $1 RETURNING *',
    [offerId]
  );
  
  if (result.rows.length === 0) {
    throw new Error('Offer not found');
  }
  
  await logAdminAction(adminUserId, 'offer_deactivated', null, {
    offer_id: offerId,
    offer_code: result.rows[0].offer_code
  }, req);
  
  return result.rows[0];
}

/**
 * Delete offer
 */
async function deleteOffer(adminUserId, offerId, req = null) {
  // Check if offer has any redemptions
  const redemptionsCheck = await db.pool.query(
    'SELECT COUNT(*) as count FROM offer_redemptions WHERE offer_id = $1',
    [offerId]
  );
  
  if (parseInt(redemptionsCheck.rows[0].count, 10) > 0) {
    throw new Error('Cannot delete offer that has been redeemed. Deactivate it instead.');
  }
  
  const result = await db.pool.query(
    'DELETE FROM promotional_offers WHERE id = $1 RETURNING *',
    [offerId]
  );
  
  if (result.rows.length === 0) {
    throw new Error('Offer not found');
  }
  
  await logAdminAction(adminUserId, 'offer_deleted', null, {
    offer_id: offerId,
    offer_code: result.rows[0].offer_code
  }, req);
  
  return result.rows[0];
}

/**
 * Get offer redemptions
 */
async function getOfferRedemptions(offerId) {
  const result = await db.pool.query(
    `SELECT or.*, u.username, u.email
     FROM offer_redemptions or
     JOIN users u ON or.user_id = u.id
     WHERE or.offer_id = $1
     ORDER BY or.redeemed_at DESC`,
    [offerId]
  );
  
  return result.rows;
}

/**
 * Get offer statistics
 */
async function getOfferStatistics(offerId) {
  const result = await db.pool.query(
    `SELECT 
       COUNT(*) as total_redemptions,
       COUNT(CASE WHEN is_active = TRUE THEN 1 END) as active_subscriptions,
       COUNT(CASE WHEN is_active = FALSE THEN 1 END) as cancelled_subscriptions
     FROM offer_redemptions
     WHERE offer_id = $1`,
    [offerId]
  );
  
  return result.rows[0];
}

/**
 * Check if user can redeem offer
 */
async function canUserRedeemOffer(userId, offerCode) {
  const offer = await getOfferByCode(offerCode);
  
  if (!offer) {
    return { canRedeem: false, reason: 'Offer code not found' };
  }
  
  if (!offer.is_active) {
    return { canRedeem: false, reason: 'Offer is no longer active' };
  }
  
  // Check validity period
  const now = new Date();
  if (offer.valid_from && new Date(offer.valid_from) > now) {
    return { canRedeem: false, reason: 'Offer has not started yet' };
  }
  
  if (offer.valid_until && new Date(offer.valid_until) < now) {
    return { canRedeem: false, reason: 'Offer has expired' };
  }
  
  // Check max redemptions
  if (offer.max_redemptions && offer.current_redemptions >= offer.max_redemptions) {
    return { canRedeem: false, reason: 'Offer has reached maximum redemptions' };
  }
  
  // Check per-user limit
  const userRedemptions = await db.pool.query(
    'SELECT COUNT(*) as count FROM offer_redemptions WHERE offer_id = $1 AND user_id = $2',
    [offer.id, userId]
  );
  
  const userCount = parseInt(userRedemptions.rows[0].count, 10);
  if (userCount >= offer.max_redemptions_per_user) {
    return { canRedeem: false, reason: 'You have already used this offer' };
  }
  
  return { canRedeem: true, offer };
}

/**
 * Record offer redemption
 */
async function recordOfferRedemption(offerId, userId, stripeSubscriptionId = null) {
  // Get offer details for snapshot
  const offer = await getOfferById(offerId);
  
  const result = await db.pool.query(
    `INSERT INTO offer_redemptions (offer_id, user_id, stripe_subscription_id, offer_details)
     VALUES ($1, $2, $3, $4)
     RETURNING *`,
    [offerId, userId, stripeSubscriptionId, JSON.stringify(offer)]
  );
  
  // Increment redemption counter
  await db.pool.query(
    'UPDATE promotional_offers SET current_redemptions = current_redemptions + 1 WHERE id = $1',
    [offerId]
  );
  
  return result.rows[0];
}

module.exports = {
  requireSuperuser,
  requireSuperuserSubdomain,
  logAdminAction,
  getAllUsers,
  getUserStatistics,
  getRevenueStatistics,
  updateUserTier,
  suspendUser,
  activateUser,
  updateUserLimits,
  deleteUser,
  getAllTiers,
  getTierById,
  createTier,
  updateTier,
  deactivateTier,
  getRecentAdminActions,
  makeSuperuser,
  removeSuperuser,
  togglePaymentEnforcement,
  createUserManually,
  // Promotional offers
  getAllOffers,
  getOfferById,
  getOfferByCode,
  createOffer,
  updateOffer,
  deactivateOffer,
  deleteOffer,
  getOfferRedemptions,
  getOfferStatistics,
  canUserRedeemOffer,
  recordOfferRedemption
};

