<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use App\Models\Colourways;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrders;
use App\Models\ShipmentLine;
use App\Models\Shipment;
use App\Services\PriceResolutionService;
use App\Services\TotalCacheService;
use Illuminate\Support\Facades\Log;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Log::info('Starting cache population migration...');
        
        $priceService = app(PriceResolutionService::class);
        $totalService = app(TotalCacheService::class);
        
        $processed = 0;
        $errors = 0;
        
        // 1. Populate Price Resolution Cache for all colourways
        Log::info('Populating price resolution cache...');
        
        $colourways = Colourways::with(['style_versions.styles.seasons', 'style_versions.factories', 'style_versions.styles.customers'])
            ->whereHas('style_versions.styles.seasons') // Only process colourways with seasons
            ->get();
            
        foreach ($colourways as $colourway) {
            try {
                $warmed = $priceService->warmupColourway($colourway->id);
                $processed++;
                
                if ($processed % 100 === 0) {
                    Log::info("Processed {$processed} colourways...");
                }
            } catch (\Exception $e) {
                $errors++;
                Log::error("Failed to warm up colourway {$colourway->id}: " . $e->getMessage());
            }
        }
        
        Log::info("Price resolution cache populated. Processed: {$processed}, Errors: {$errors}");
        
        // 2. Populate Total Cache for Customer Order Lines
        Log::info('Populating customer order line totals cache...');
        
        $processed = 0;
        $errors = 0;
        
        $orderLines = CustomerOrderLines::with([
            'customer_order_line_quantities.price_model',
            'colourways.style_versions.styles.seasons'
        ])->get();
        
        foreach ($orderLines as $orderLine) {
            try {
                $totalService->warmupCustomerOrderLine($orderLine);
                $processed++;
                
                if ($processed % 100 === 0) {
                    Log::info("Processed {$processed} customer order lines...");
                }
            } catch (\Exception $e) {
                $errors++;
                Log::error("Failed to warm up customer order line {$orderLine->id}: " . $e->getMessage());
            }
        }
        
        Log::info("Customer order line totals cache populated. Processed: {$processed}, Errors: {$errors}");
        
        // 3. Populate Total Cache for Customer Orders
        Log::info('Populating customer order totals cache...');
        
        $processed = 0;
        $errors = 0;
        
        $orders = CustomerOrders::with(['customer_order_lines.customer_order_line_quantities.price_model'])->get();
        
        foreach ($orders as $order) {
            try {
                $totalService->warmupCustomerOrder($order);
                $processed++;
                
                if ($processed % 50 === 0) {
                    Log::info("Processed {$processed} customer orders...");
                }
            } catch (\Exception $e) {
                $errors++;
                Log::error("Failed to warm up customer order {$order->id}: " . $e->getMessage());
            }
        }
        
        Log::info("Customer order totals cache populated. Processed: {$processed}, Errors: {$errors}");
        
        // 4. Populate Total Cache for Shipment Lines
        Log::info('Populating shipment line totals cache...');
        
        $processed = 0;
        $errors = 0;
        
        $shipmentLines = ShipmentLine::with([
            'customer_order_lines.customer_order_line_quantities.price_model',
            'shipment_line_sizes'
        ])->get();
        
        foreach ($shipmentLines as $shipmentLine) {
            try {
                $totalService->warmupShipmentLine($shipmentLine);
                $processed++;
                
                if ($processed % 100 === 0) {
                    Log::info("Processed {$processed} shipment lines...");
                }
            } catch (\Exception $e) {
                $errors++;
                Log::error("Failed to warm up shipment line {$shipmentLine->id}: " . $e->getMessage());
            }
        }
        
        Log::info("Shipment line totals cache populated. Processed: {$processed}, Errors: {$errors}");
        
        // 5. Populate Total Cache for Shipments
        Log::info('Populating shipment totals cache...');
        
        $processed = 0;
        $errors = 0;
        
        $shipments = Shipment::with([
            'shipment_lines.customer_order_lines.customer_order_line_quantities.price_model',
            'shipment_lines.shipment_line_sizes'
        ])->get();
        
        foreach ($shipments as $shipment) {
            try {
                $totalService->warmupShipment($shipment);
                $processed++;
                
                if ($processed % 50 === 0) {
                    Log::info("Processed {$processed} shipments...");
                }
            } catch (\Exception $e) {
                $errors++;
                Log::error("Failed to warm up shipment {$shipment->id}: " . $e->getMessage());
            }
        }
        
        Log::info("Shipment totals cache populated. Processed: {$processed}, Errors: {$errors}");
        
        // 6. Log final statistics
        $priceStats = $priceService->getCacheStats();
        $totalStats = $totalService->getCacheStats();
        
        Log::info('Cache population completed!', [
            'price_resolutions' => $priceStats,
            'total_cache' => $totalStats
        ]);
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Clear all caches
        Log::info('Clearing all caches...');
        
        // Clear price resolution cache
        \App\Models\PriceResolution::truncate();
        
        // Clear total cache
        \App\Models\TotalCache::truncate();
        
        // Clear Laravel cache
        \Illuminate\Support\Facades\Cache::flush();
        
        Log::info('All caches cleared.');
    }
};