<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use App\Models\Colourways;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrders;
use App\Models\ShipmentLine;
use App\Models\Shipment;
use App\Services\PriceResolutionService;
use App\Services\TotalCacheService;
use Illuminate\Support\Facades\Log;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Log::info('Starting cache population migration...');
        
        $priceService = app(PriceResolutionService::class);
        $totalService = app(TotalCacheService::class);
        
        // 1. Populate Price Resolution Cache for all colourways (in chunks)
        Log::info('Populating price resolution cache...');
        
        $chunkSize = 50; // Process in smaller chunks
        $processed = 0;
        $errors = 0;
        
        Colourways::with(['style_versions.styles.seasons', 'style_versions.factories', 'style_versions.styles.customers'])
            ->whereHas('style_versions.styles.seasons')
            ->chunk($chunkSize, function ($colourways) use ($priceService, &$processed, &$errors) {
                foreach ($colourways as $colourway) {
                    try {
                        $priceService->warmupColourway($colourway->id);
                        $processed++;
                        
                        if ($processed % 100 === 0) {
                            Log::info("Processed {$processed} colourways...");
                        }
                    } catch (\Exception $e) {
                        $errors++;
                        Log::error("Failed to warm up colourway {$colourway->id}: " . $e->getMessage());
                    }
                }
            });
        
        Log::info("Price resolution cache populated. Processed: {$processed}, Errors: {$errors}");
        
        // 2. Populate Total Cache for Customer Order Lines (in chunks)
        Log::info('Populating customer order line totals cache...');
        
        $processed = 0;
        $errors = 0;
        
        CustomerOrderLines::chunk($chunkSize, function ($orderLines) use ($totalService, &$processed, &$errors) {
            foreach ($orderLines as $orderLine) {
                try {
                    $totalService->warmupCustomerOrderLine($orderLine);
                    $processed++;
                    
                    if ($processed % 100 === 0) {
                        Log::info("Processed {$processed} customer order lines...");
                    }
                } catch (\Exception $e) {
                    $errors++;
                    Log::error("Failed to warm up customer order line {$orderLine->id}: " . $e->getMessage());
                }
            }
        });
        
        Log::info("Customer order line totals cache populated. Processed: {$processed}, Errors: {$errors}");
        
        // 3. Populate Total Cache for Customer Orders (in chunks)
        Log::info('Populating customer order totals cache...');
        
        $processed = 0;
        $errors = 0;
        
        CustomerOrders::chunk($chunkSize, function ($orders) use ($totalService, &$processed, &$errors) {
            foreach ($orders as $order) {
                try {
                    $totalService->warmupCustomerOrder($order);
                    $processed++;
                    
                    if ($processed % 50 === 0) {
                        Log::info("Processed {$processed} customer orders...");
                    }
                } catch (\Exception $e) {
                    $errors++;
                    Log::error("Failed to warm up customer order {$order->id}: " . $e->getMessage());
                }
            }
        });
        
        Log::info("Customer order totals cache populated. Processed: {$processed}, Errors: {$errors}");
        
        // 4. Populate Total Cache for Shipment Lines (in chunks)
        Log::info('Populating shipment line totals cache...');
        
        $processed = 0;
        $errors = 0;
        
        ShipmentLine::chunk($chunkSize, function ($shipmentLines) use ($totalService, &$processed, &$errors) {
            foreach ($shipmentLines as $shipmentLine) {
                try {
                    $totalService->warmupShipmentLine($shipmentLine);
                    $processed++;
                    
                    if ($processed % 100 === 0) {
                        Log::info("Processed {$processed} shipment lines...");
                    }
                } catch (\Exception $e) {
                    $errors++;
                    Log::error("Failed to warm up shipment line {$shipmentLine->id}: " . $e->getMessage());
                }
            }
        });
        
        Log::info("Shipment line totals cache populated. Processed: {$processed}, Errors: {$errors}");
        
        // 5. Populate Total Cache for Shipments (in chunks)
        Log::info('Populating shipment totals cache...');
        
        $processed = 0;
        $errors = 0;
        
        Shipment::chunk($chunkSize, function ($shipments) use ($totalService, &$processed, &$errors) {
            foreach ($shipments as $shipment) {
                try {
                    $totalService->warmupShipment($shipment);
                    $processed++;
                    
                    if ($processed % 50 === 0) {
                        Log::info("Processed {$processed} shipments...");
                    }
                } catch (\Exception $e) {
                    $errors++;
                    Log::error("Failed to warm up shipment {$shipment->id}: " . $e->getMessage());
                }
            }
        });
        
        Log::info("Shipment totals cache populated. Processed: {$processed}, Errors: {$errors}");
        
        // 6. Log final statistics
        $priceStats = $priceService->getCacheStats();
        $totalStats = $totalService->getCacheStats();
        
        Log::info('Cache population completed!', [
            'price_resolutions' => $priceStats,
            'total_cache' => $totalStats
        ]);
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Clear all caches
        Log::info('Clearing all caches...');
        
        // Clear price resolution cache
        \App\Models\PriceResolution::truncate();
        
        // Clear total cache
        \App\Models\TotalCache::truncate();
        
        // Clear Laravel cache
        \Illuminate\Support\Facades\Cache::flush();
        
        Log::info('All caches cleared.');
    }
};