<?php

namespace App\Livewire\Admin;

use App\Models\User;
use Livewire\Component;
use Livewire\WithPagination;

class CustomerDetail extends Component
{
    use WithPagination;
    
    public User $customer;
    public $editing = false;
    
    // Form fields for editing
    public $name;
    public $email;
    public $role;
    
    protected $rules = [
        'name' => 'required|string|max:255',
        'email' => 'required|email|max:255',
        'role' => 'required|in:user,admin',
    ];
    
    public function mount($customerId)
    {
        $this->customer = User::findOrFail($customerId);
        $this->loadFormData();
    }
    
    public function loadFormData()
    {
        $this->name = $this->customer->name;
        $this->email = $this->customer->email;
        $this->role = $this->customer->role;
    }
    
    public function toggleEdit()
    {
        if ($this->editing) {
            $this->loadFormData(); // Reset form data
        }
        $this->editing = !$this->editing;
    }
    
    public function save()
    {
        $this->validate();
        
        // Check if email is unique (excluding current customer)
        $existingUser = User::where('email', $this->email)
            ->where('id', '!=', $this->customer->id)
            ->first();
            
        if ($existingUser) {
            $this->addError('email', 'This email is already taken by another user.');
            return;
        }
        
        $this->customer->update([
            'name' => $this->name,
            'email' => $this->email,
            'role' => $this->role,
        ]);
        
        $this->editing = false;
        session()->flash('message', 'Customer updated successfully!');
    }
    
    public function cancel()
    {
        $this->loadFormData();
        $this->editing = false;
    }
    
    public function render()
    {
        // TODO: Add orders when Order model is created
        $orders = collect(); // Empty collection for now
            
        return view('livewire.admin.customer-detail', [
            'orders' => $orders
        ])
            ->layout('layouts.admin')
            ->title('Customer Details');
    }
}
