<?php

namespace App\Livewire\Admin;

use App\Models\User;
use Livewire\Component;
use Livewire\WithPagination;

class CustomerDetail extends Component
{
    use WithPagination;
    
    public User $customer;
    public $editing = false;
    
    // Form fields for editing
    public $name;
    public $email;
    public $role;
    
    // Address management
    public $showAddressForm = false;
    public $editingAddress = null;
    public $addressType = 'billing';
    public $addressLine1 = '';
    public $addressLine2 = '';
    public $city = '';
    public $state = '';
    public $postalCode = '';
    public $country = '';
    public $isDefault = false;
    
    // Temporary storage for addresses (until Address model is created)
    public $tempAddresses = [];
    
    protected $rules = [
        'name' => 'required|string|max:255',
        'email' => 'required|email|max:255',
        'role' => 'required|in:user,admin',
        'addressType' => 'required|in:billing,shipping',
        'addressLine1' => 'required|string|max:255',
        'addressLine2' => 'nullable|string|max:255',
        'city' => 'required|string|max:100',
        'state' => 'required|string|max:100',
        'postalCode' => 'required|string|max:20',
        'country' => 'required|string|max:100',
        'isDefault' => 'boolean',
    ];
    
    public function mount($customerId)
    {
        $this->customer = User::findOrFail($customerId);
        $this->loadFormData();
    }
    
    public function loadFormData()
    {
        $this->name = $this->customer->name;
        $this->email = $this->customer->email;
        $this->role = $this->customer->role;
    }
    
    public function toggleEdit()
    {
        if ($this->editing) {
            $this->loadFormData(); // Reset form data
        }
        $this->editing = !$this->editing;
    }
    
    public function save()
    {
        $this->validate();
        
        // Check if email is unique (excluding current customer)
        $existingUser = User::where('email', $this->email)
            ->where('id', '!=', $this->customer->id)
            ->first();
            
        if ($existingUser) {
            $this->addError('email', 'This email is already taken by another user.');
            return;
        }
        
        $this->customer->update([
            'name' => $this->name,
            'email' => $this->email,
            'role' => $this->role,
        ]);
        
        $this->editing = false;
        session()->flash('message', 'Customer updated successfully!');
    }
    
    public function cancel()
    {
        $this->loadFormData();
        $this->editing = false;
    }
    
    // Address management methods
    public function showAddAddressForm()
    {
        $this->resetAddressForm();
        $this->showAddressForm = true;
        $this->editingAddress = null;
    }
    
    public function editAddress($addressId)
    {
        // TODO: Load address data when Address model is created
        $this->showAddressForm = true;
        $this->editingAddress = $addressId;
        $this->resetAddressForm();
    }
    
    public function saveAddress()
    {
        $this->validate([
            'addressType' => 'required|in:billing,shipping',
            'addressLine1' => 'required|string|max:255',
            'addressLine2' => 'nullable|string|max:255',
            'city' => 'required|string|max:100',
            'state' => 'required|string|max:100',
            'postalCode' => 'required|string|max:20',
            'country' => 'required|string|max:100',
            'isDefault' => 'boolean',
        ]);
        
        // TODO: Save address when Address model is created
        // For now, just show success message
        session()->flash('message', 'Address saved successfully!');
        $this->resetAddressForm();
        $this->showAddressForm = false;
    }
    
    public function deleteAddress($addressId)
    {
        // TODO: Delete address when Address model is created
        session()->flash('message', 'Address deleted successfully!');
    }
    
    public function cancelAddressForm()
    {
        $this->resetAddressForm();
        $this->showAddressForm = false;
        $this->editingAddress = null;
    }
    
    public function resetAddressForm()
    {
        $this->addressType = 'billing';
        $this->addressLine1 = '';
        $this->addressLine2 = '';
        $this->city = '';
        $this->state = '';
        $this->postalCode = '';
        $this->country = '';
        $this->isDefault = false;
        $this->resetErrorBag();
    }
    
    public function render()
    {
        // TODO: Add orders when Order model is created
        $orders = collect(); // Empty collection for now
        
        // TODO: Add addresses when Address model is created
        $addresses = collect(); // Empty collection for now
            
        return view('livewire.admin.customer-detail', [
            'orders' => $orders,
            'addresses' => $addresses
        ])
            ->layout('layouts.admin')
            ->title('Customer Details');
    }
}
