<?php

namespace App\Livewire\Admin;

use App\Models\User;
use App\Models\Address;
use Livewire\Component;
use Livewire\WithPagination;

class CustomerDetail extends Component
{
    use WithPagination;
    
    public User $customer;
    public $editing = false;
    
    // Form fields for editing
    public $name;
    public $email;
    public $role;
    public $marketingConsent;
    
    // Address management
    public $showAddressForm = false;
    public $editingAddress = null;
    public $addressType = 'shipping';
    public $firstName = '';
    public $lastName = '';
    public $company = '';
    public $addressLine1 = '';
    public $addressLine2 = '';
    public $city = '';
    public $stateProvince = '';
    public $postalCode = '';
    public $country = '';
    public $phone = '';
    public $isDefault = false;
    
    protected $rules = [
        'name' => 'required|string|max:255',
        'email' => 'required|email|max:255',
        'role' => 'required|in:user,admin',
        'addressType' => 'required|string|max:255',
        'firstName' => 'required|string|max:255',
        'lastName' => 'required|string|max:255',
        'company' => 'nullable|string|max:255',
        'addressLine1' => 'required|string|max:255',
        'addressLine2' => 'nullable|string|max:255',
        'city' => 'required|string|max:100',
        'stateProvince' => 'nullable|string|max:100',
        'postalCode' => 'required|string|max:20',
        'country' => 'required|string|max:100',
        'phone' => 'nullable|string|max:20',
        'isDefault' => 'boolean',
    ];
    
    public function mount($customerId)
    {
        $this->customer = User::findOrFail($customerId);
        $this->loadFormData();
    }
    
    public function loadFormData()
    {
        $this->name = $this->customer->name;
        $this->email = $this->customer->email;
        $this->role = $this->customer->role;
    }
    
    public function toggleEdit()
    {
        if ($this->editing) {
            $this->loadFormData(); // Reset form data
        }
        $this->editing = !$this->editing;
    }
    
    public function save()
    {
        $this->validate();
        
        // Check if email is unique (excluding current customer)
        $existingUser = User::where('email', $this->email)
            ->where('id', '!=', $this->customer->id)
            ->first();
            
        if ($existingUser) {
            $this->addError('email', 'This email is already taken by another user.');
            return;
        }
        
        $this->customer->update([
            'name' => $this->name,
            'email' => $this->email,
            'role' => $this->role,
        ]);
        
        $this->editing = false;
        session()->flash('message', 'Customer updated successfully!');
    }
    
    public function cancel()
    {
        $this->loadFormData();
        $this->editing = false;
    }
    
    // Address management methods
    public function showAddAddressForm()
    {
        $this->resetAddressForm();
        $this->showAddressForm = true;
        $this->editingAddress = null;
    }
    
    public function editAddress($addressId)
    {
        $address = Address::find($addressId);
        
        if ($address) {
            $this->editingAddress = $addressId;
            $this->addressType = $address->type;
            $this->firstName = $address->first_name;
            $this->lastName = $address->last_name;
            $this->company = $address->company;
            $this->addressLine1 = $address->address_line_1;
            $this->addressLine2 = $address->address_line_2;
            $this->city = $address->city;
            $this->stateProvince = $address->state_province;
            $this->postalCode = $address->postal_code;
            $this->country = $address->country;
            $this->phone = $address->phone;
            $this->isDefault = $address->is_default;
            $this->showAddressForm = true;
        }
    }
    
    public function saveAddress()
    {
        $this->validate([
            'addressType' => 'required|string|max:255',
            'firstName' => 'required|string|max:255',
            'lastName' => 'required|string|max:255',
            'company' => 'nullable|string|max:255',
            'addressLine1' => 'required|string|max:255',
            'addressLine2' => 'nullable|string|max:255',
            'city' => 'required|string|max:100',
            'stateProvince' => 'nullable|string|max:100',
            'postalCode' => 'required|string|max:20',
            'country' => 'required|string|max:100',
            'phone' => 'nullable|string|max:20',
            'isDefault' => 'boolean',
        ]);
        
        // If setting as default, remove default from other addresses of same type
        if ($this->isDefault) {
            Address::where('users_id', $this->customer->id)
                  ->where('type', $this->addressType)
                  ->update(['is_default' => false]);
        }
        
        $addressData = [
            'users_id' => $this->customer->id,
            'type' => $this->addressType,
            'first_name' => $this->firstName,
            'last_name' => $this->lastName,
            'company' => $this->company,
            'address_line_1' => $this->addressLine1,
            'address_line_2' => $this->addressLine2,
            'city' => $this->city,
            'state_province' => $this->stateProvince,
            'postal_code' => $this->postalCode,
            'country' => $this->country,
            'phone' => $this->phone,
            'is_default' => $this->isDefault,
        ];
        
        if ($this->editingAddress) {
            Address::where('id', $this->editingAddress)->update($addressData);
        } else {
            Address::create($addressData);
        }
        
        session()->flash('message', 'Address saved successfully!');
        $this->resetAddressForm();
        $this->showAddressForm = false;
    }
    
    public function deleteAddress($addressId)
    {
        Address::where('id', $addressId)->delete();
        session()->flash('message', 'Address deleted successfully!');
    }
    
    public function cancelAddressForm()
    {
        $this->resetAddressForm();
        $this->showAddressForm = false;
        $this->editingAddress = null;
    }
    
    public function resetAddressForm()
    {
        $this->addressType = 'shipping';
        $this->firstName = '';
        $this->lastName = '';
        $this->company = '';
        $this->addressLine1 = '';
        $this->addressLine2 = '';
        $this->city = '';
        $this->stateProvince = '';
        $this->postalCode = '';
        $this->country = '';
        $this->phone = '';
        $this->isDefault = false;
        $this->resetErrorBag();
    }
    
    public function render()
    {
        // TODO: Add orders when Order model is created
        $orders = collect(); // Empty collection for now
        
        // Get addresses from database
        $addresses = $this->customer->addresses;
            
        return view('livewire.admin.customer-detail', [
            'orders' => $orders,
            'addresses' => $addresses
        ])
            ->layout('layouts.admin')
            ->title('Customer Details');
    }
}
