<?php

namespace App\Console\Commands;

use App\Models\Receipt;
use App\Models\ReceiptGroup;
use App\Services\AI\ReceiptGroupingService;
use Illuminate\Console\Command;

class TestReceiptGrouping extends Command
{
    protected $signature = 'test:receipt-grouping';
    protected $description = 'Test receipt grouping functionality';

    public function handle()
    {
        $this->info('Testing Receipt Grouping Functionality...');
        
        // Get some unmatched receipts
        $unmatchedReceipts = Receipt::whereNull('receipt_group_id')
            ->whereDoesntHave('matches')
            ->with(['lines'])
            ->take(10)
            ->get();

        if ($unmatchedReceipts->count() < 2) {
            $this->warn('Not enough unmatched receipts to test grouping. Need at least 2 receipts.');
            return;
        }

        $this->info("Found {$unmatchedReceipts->count()} unmatched receipts");

        // Test AI suggestions
        $groupingService = app(ReceiptGroupingService::class);
        $suggestions = $groupingService->suggestGroupings($unmatchedReceipts);

        $this->info("AI generated " . count($suggestions) . " grouping suggestions:");

        foreach ($suggestions as $index => $suggestion) {
            $this->line("Suggestion " . ($index + 1) . ":");
            $this->line("  Group Name: " . ($suggestion['group_name'] ?? 'Unnamed'));
            $this->line("  Receipt IDs: " . implode(', ', $suggestion['receipt_ids'] ?? []));
            $this->line("  Confidence: " . round(($suggestion['confidence'] ?? 0) * 100) . "%");
            $this->line("  Reasoning: " . ($suggestion['reasoning'] ?? 'No reasoning'));
            $this->line("");
        }

        // Show existing groups
        $existingGroups = ReceiptGroup::with(['receipts'])->get();
        $this->info("Existing receipt groups: " . $existingGroups->count());

        foreach ($existingGroups as $group) {
            $this->line("Group: {$group->name} ({$group->receipts->count()} receipts)");
            $this->line("  Method: {$group->grouping_method}");
            $this->line("  Total Amount: £" . number_format($group->total_amount, 2));
            $this->line("  Primary Merchant: " . ($group->primary_merchant ?? 'N/A'));
            $this->line("");
        }

        $this->info('Receipt grouping test completed!');
    }
}