const stripeService = require('./stripe-service');

/**
 * Handle Stripe webhook events
 * This function processes various Stripe events and updates the database accordingly
 */
async function handleWebhook(db, event) {
  console.log('[STRIPE WEBHOOK]', event.type, event.id);
  
  try {
    switch (event.type) {
      // Subscription lifecycle events
      case 'customer.subscription.created':
        await handleSubscriptionCreated(db, event.data.object);
        break;
        
      case 'customer.subscription.updated':
        await handleSubscriptionUpdated(db, event.data.object);
        break;
        
      case 'customer.subscription.deleted':
        await handleSubscriptionDeleted(db, event.data.object);
        break;
        
      // Trial events
      case 'customer.subscription.trial_will_end':
        await handleTrialWillEnd(db, event.data.object);
        break;
        
      // Payment events
      case 'invoice.paid':
        await handleInvoicePaid(db, event.data.object);
        break;
        
      case 'invoice.payment_failed':
        await handleInvoicePaymentFailed(db, event.data.object);
        break;
        
      // Customer events
      case 'customer.created':
        await handleCustomerCreated(db, event.data.object);
        break;
        
      case 'customer.updated':
        await handleCustomerUpdated(db, event.data.object);
        break;
        
      default:
        console.log('[STRIPE WEBHOOK] Unhandled event type:', event.type);
    }
    
    // Log the event to the database
    await logBillingEvent(db, event);
    
    return { received: true };
  } catch (error) {
    console.error('[STRIPE WEBHOOK] Error processing webhook:', error);
    throw error;
  }
}

/**
 * Handle subscription.created event
 */
async function handleSubscriptionCreated(db, subscription) {
  const customerId = subscription.customer;
  const subscriptionId = subscription.id;
  const tier = subscription.metadata.tier || 'starter';
  const status = mapStripeStatus(subscription.status);
  const trialEnd = subscription.trial_end ? new Date(subscription.trial_end * 1000) : null;
  const currentPeriodEnd = new Date(subscription.current_period_end * 1000);
  
  console.log('[STRIPE] Subscription created:', {
    customerId,
    subscriptionId,
    tier,
    status,
    trialEnd,
  });
  
  // Update user with subscription details
  await db.updateUserByStripeCustomerId(customerId, {
    stripeSubscriptionId: subscriptionId,
    subscriptionTier: tier,
    subscriptionStatus: status,
    trialEndsAt: trialEnd,
    subscriptionEndsAt: currentPeriodEnd,
  });
}

/**
 * Handle subscription.updated event
 */
async function handleSubscriptionUpdated(db, subscription) {
  const customerId = subscription.customer;
  const subscriptionId = subscription.id;
  const tier = subscription.metadata.tier || 'starter';
  const status = mapStripeStatus(subscription.status);
  const trialEnd = subscription.trial_end ? new Date(subscription.trial_end * 1000) : null;
  const currentPeriodEnd = new Date(subscription.current_period_end * 1000);
  const cancelAt = subscription.cancel_at ? new Date(subscription.cancel_at * 1000) : null;
  
  console.log('[STRIPE] Subscription updated:', {
    customerId,
    subscriptionId,
    tier,
    status,
    cancelAt,
  });
  
  // Update user with new subscription details
  await db.updateUserByStripeCustomerId(customerId, {
    stripeSubscriptionId: subscriptionId,
    subscriptionTier: tier,
    subscriptionStatus: status,
    trialEndsAt: trialEnd,
    subscriptionEndsAt: cancelAt || currentPeriodEnd,
  });
}

/**
 * Handle subscription.deleted event
 */
async function handleSubscriptionDeleted(db, subscription) {
  const customerId = subscription.customer;
  const subscriptionId = subscription.id;
  
  console.log('[STRIPE] Subscription deleted:', {
    customerId,
    subscriptionId,
  });
  
  // Downgrade user to free tier
  await db.updateUserByStripeCustomerId(customerId, {
    subscriptionTier: 'free',
    subscriptionStatus: 'canceled',
    stripeSubscriptionId: null,
  });
}

/**
 * Handle trial_will_end event (3 days before trial ends)
 */
async function handleTrialWillEnd(db, subscription) {
  const customerId = subscription.customer;
  const trialEnd = new Date(subscription.trial_end * 1000);
  
  console.log('[STRIPE] Trial will end soon:', {
    customerId,
    trialEnd,
  });
  
  // TODO: Send email notification to user
  // You can implement email notifications here
}

/**
 * Handle invoice.paid event
 */
async function handleInvoicePaid(db, invoice) {
  const customerId = invoice.customer;
  const subscriptionId = invoice.subscription;
  
  console.log('[STRIPE] Invoice paid:', {
    customerId,
    subscriptionId,
    amount: invoice.amount_paid,
  });
  
  // If subscription exists, ensure user is marked as active
  if (subscriptionId) {
    await db.updateUserByStripeCustomerId(customerId, {
      subscriptionStatus: 'active',
    });
  }
}

/**
 * Handle invoice.payment_failed event
 */
async function handleInvoicePaymentFailed(db, invoice) {
  const customerId = invoice.customer;
  const subscriptionId = invoice.subscription;
  
  console.log('[STRIPE] Invoice payment failed:', {
    customerId,
    subscriptionId,
    amount: invoice.amount_due,
  });
  
  // Mark subscription as past_due
  if (subscriptionId) {
    await db.updateUserByStripeCustomerId(customerId, {
      subscriptionStatus: 'past_due',
    });
  }
  
  // TODO: Send email notification to user about failed payment
}

/**
 * Handle customer.created event
 */
async function handleCustomerCreated(db, customer) {
  console.log('[STRIPE] Customer created:', {
    customerId: customer.id,
    email: customer.email,
  });
  
  // Customer should already exist in database from signup
  // This event is mainly for logging
}

/**
 * Handle customer.updated event
 */
async function handleCustomerUpdated(db, customer) {
  console.log('[STRIPE] Customer updated:', {
    customerId: customer.id,
    email: customer.email,
  });
  
  // Update user email if changed
  if (customer.email) {
    await db.updateUserByStripeCustomerId(customer.id, {
      email: customer.email,
    });
  }
}

/**
 * Map Stripe subscription status to our internal status
 */
function mapStripeStatus(stripeStatus) {
  const statusMap = {
    'incomplete': 'trialing',
    'incomplete_expired': 'canceled',
    'trialing': 'trialing',
    'active': 'active',
    'past_due': 'past_due',
    'canceled': 'canceled',
    'unpaid': 'past_due',
    'paused': 'paused',
  };
  
  return statusMap[stripeStatus] || 'canceled';
}

/**
 * Log billing event to database
 */
async function logBillingEvent(db, event) {
  try {
    // Extract customer ID from event if available
    let customerId = null;
    if (event.data.object.customer) {
      customerId = event.data.object.customer;
    }
    
    // Get user ID from customer ID
    let userId = null;
    if (customerId) {
      const user = await db.getUserByStripeCustomerId(customerId);
      userId = user?.id || null;
    }
    
    await db.logBillingEvent({
      userId,
      stripeEventId: event.id,
      eventType: event.type,
      eventData: event,
      processed: true,
    });
  } catch (error) {
    console.error('[STRIPE WEBHOOK] Error logging billing event:', error);
  }
}

module.exports = {
  handleWebhook,
};

