<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Customer extends Model
{
    protected $fillable = [
        'first_name',
        'last_name',
        'email',
        'phone',
        'notes',
        'marketing_opt_in',
    ];

    protected $casts = [
        'marketing_opt_in' => 'boolean',
    ];

    /**
     * Get the addresses for the customer.
     */
    public function addresses(): HasMany
    {
        return $this->hasMany(Address::class);
    }

    /**
     * Get the orders for the customer.
     */
    public function orders(): HasMany
    {
        return $this->hasMany(Order::class);
    }

    /**
     * Get the default billing address for the customer.
     */
    public function defaultBillingAddress()
    {
        return $this->addresses()->where('is_default_billing', true)->first();
    }

    /**
     * Get the default shipping address for the customer.
     */
    public function defaultShippingAddress()
    {
        return $this->addresses()->where('is_default_shipping', true)->first();
    }

    /**
     * Scope a query to only include customers with marketing opt-in.
     */
    public function scopeMarketingOptIn($query)
    {
        return $query->where('marketing_opt_in', true);
    }

    /**
     * Get the customer's full name.
     */
    public function getFullNameAttribute(): string
    {
        return trim($this->first_name . ' ' . $this->last_name);
    }
}
