-- Migration: Add Superuser Support
-- This migration adds superuser functionality for platform administration

-- Add is_superuser column to users table
ALTER TABLE users ADD COLUMN IF NOT EXISTS is_superuser BOOLEAN DEFAULT FALSE;

-- Create index for superuser lookup
CREATE INDEX IF NOT EXISTS idx_users_is_superuser ON users(is_superuser) WHERE is_superuser = TRUE;

-- Create subscription_tiers table for dynamic tier management
CREATE TABLE IF NOT EXISTS subscription_tiers (
  id SERIAL PRIMARY KEY,
  tier_key TEXT NOT NULL UNIQUE,
  tier_name TEXT NOT NULL,
  price_pence INTEGER NOT NULL,
  max_photos INTEGER NOT NULL,
  max_storage_bytes BIGINT NOT NULL,
  allows_custom_domain BOOLEAN DEFAULT FALSE,
  allows_multiple_domains BOOLEAN DEFAULT FALSE,
  display_order INTEGER DEFAULT 0,
  is_active BOOLEAN DEFAULT TRUE,
  description TEXT,
  features JSONB DEFAULT '[]',
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
);

-- Insert default subscription tiers
INSERT INTO subscription_tiers (tier_key, tier_name, price_pence, max_photos, max_storage_bytes, allows_custom_domain, allows_multiple_domains, display_order, description, features)
VALUES 
  ('starter', 'Starter', 1000, 1000, 5368709120, FALSE, FALSE, 1, 'Perfect for getting started with your photo gallery', '["1,000 photos", "5 GB storage", "Subdomain hosting", "Custom branding", "Unlimited albums", "Private & public galleries"]'),
  ('professional', 'Professional', 2500, 10000, 53687091200, TRUE, FALSE, 2, 'For professional photographers with growing businesses', '["10,000 photos", "50 GB storage", "Custom domain support", "Everything in Starter", "Priority support", "Advanced analytics"]'),
  ('business', 'Business', 7500, 30000, 161061273600, TRUE, TRUE, 3, 'For photography studios and agencies', '["30,000 photos", "150 GB storage", "Multiple custom domains", "Everything in Professional", "White-label support", "Team collaboration"]')
ON CONFLICT (tier_key) DO NOTHING;

-- Create indexes for subscription_tiers table
CREATE INDEX IF NOT EXISTS idx_subscription_tiers_tier_key ON subscription_tiers(tier_key);
CREATE INDEX IF NOT EXISTS idx_subscription_tiers_is_active ON subscription_tiers(is_active);
CREATE INDEX IF NOT EXISTS idx_subscription_tiers_display_order ON subscription_tiers(display_order);

-- Create admin_audit_log table for tracking superuser actions
CREATE TABLE IF NOT EXISTS admin_audit_log (
  id SERIAL PRIMARY KEY,
  admin_user_id INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE,
  action_type TEXT NOT NULL CHECK (action_type IN (
    'user_created', 'user_updated', 'user_deleted', 'user_suspended', 'user_activated',
    'tier_created', 'tier_updated', 'tier_deleted', 'tier_assigned',
    'subscription_modified', 'limits_adjusted', 'password_reset',
    'settings_changed', 'data_export', 'custom_domain_verified'
  )),
  target_user_id INTEGER REFERENCES users(id) ON DELETE SET NULL,
  action_details JSONB,
  ip_address TEXT,
  user_agent TEXT,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
);

-- Create indexes for admin_audit_log table
CREATE INDEX IF NOT EXISTS idx_admin_audit_log_admin_user_id ON admin_audit_log(admin_user_id);
CREATE INDEX IF NOT EXISTS idx_admin_audit_log_target_user_id ON admin_audit_log(target_user_id);
CREATE INDEX IF NOT EXISTS idx_admin_audit_log_action_type ON admin_audit_log(action_type);
CREATE INDEX IF NOT EXISTS idx_admin_audit_log_created_at ON admin_audit_log(created_at);

-- Add suspended status to users table
ALTER TABLE users DROP CONSTRAINT IF EXISTS users_subscription_status_check;
ALTER TABLE users ADD CONSTRAINT users_subscription_status_check 
  CHECK (subscription_status IN ('trialing', 'active', 'past_due', 'canceled', 'paused', 'suspended'));

-- Add is_suspended column for easier querying
ALTER TABLE users ADD COLUMN IF NOT EXISTS is_suspended BOOLEAN DEFAULT FALSE;

-- Function to update subscription_tiers updated_at timestamp
CREATE OR REPLACE FUNCTION update_subscription_tiers_updated_at()
RETURNS TRIGGER AS $$
BEGIN
  NEW.updated_at = CURRENT_TIMESTAMP;
  RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Create trigger for subscription_tiers updated_at
DROP TRIGGER IF EXISTS trigger_subscription_tiers_updated_at ON subscription_tiers;
CREATE TRIGGER trigger_subscription_tiers_updated_at
BEFORE UPDATE ON subscription_tiers
FOR EACH ROW EXECUTE FUNCTION update_subscription_tiers_updated_at();

-- Update the apply_subscription_limits function to use subscription_tiers table
CREATE OR REPLACE FUNCTION apply_subscription_limits()
RETURNS TRIGGER AS $$
DECLARE
  tier_info RECORD;
BEGIN
  -- Get limits from subscription_tiers table
  SELECT max_photos, max_storage_bytes
  INTO tier_info
  FROM subscription_tiers
  WHERE tier_key = NEW.subscription_tier AND is_active = TRUE;
  
  -- If tier found in table, use those limits
  IF FOUND THEN
    NEW.max_photos := tier_info.max_photos;
    NEW.max_storage_bytes := tier_info.max_storage_bytes;
  ELSE
    -- Fallback to default limits (backwards compatibility)
    IF NEW.subscription_tier = 'starter' THEN
      NEW.max_photos := 1000;
      NEW.max_storage_bytes := 5368709120;
    ELSIF NEW.subscription_tier = 'professional' THEN
      NEW.max_photos := 10000;
      NEW.max_storage_bytes := 53687091200;
    ELSIF NEW.subscription_tier = 'business' THEN
      NEW.max_photos := 30000;
      NEW.max_storage_bytes := 161061273600;
    END IF;
  END IF;
  
  NEW.updated_at := CURRENT_TIMESTAMP;
  RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Comments
COMMENT ON TABLE subscription_tiers IS 'Configurable subscription tier definitions';
COMMENT ON TABLE admin_audit_log IS 'Audit log for superuser administrative actions';
COMMENT ON COLUMN users.is_superuser IS 'Whether this user has superuser/admin privileges';
COMMENT ON COLUMN users.is_suspended IS 'Whether this user account is suspended by admin';

