<?php

namespace App\Services;

use App\Models\Receipt;
use App\Models\Statement;
use App\Models\StatementTransaction;
use Illuminate\Support\Carbon;

class MatchingService
{
    public function attemptMatchForReceipt(Receipt $receipt): void
    {
        // Find open statements that cover or are near the receipt date
        $windowStart = Carbon::parse($receipt->receipt_date)->subDays(5);
        $windowEnd = Carbon::parse($receipt->receipt_date)->addDays(7);

        $candidateTransactions = StatementTransaction::query()
            ->whereHas('statement', fn ($q) => $q->where('status', 'open')
                ->whereDate('start_date', '<=', $windowEnd)
                ->whereDate('end_date', '>=', $windowStart))
            ->whereBetween('transaction_date', [$windowStart, $windowEnd])
            ->whereBetween('amount', [
                (float)$receipt->total_amount - 1.00,
                (float)$receipt->total_amount + 1.00,
            ])
            ->orderByRaw('ABS(DATEDIFF(transaction_date, ?)) asc', [$receipt->receipt_date])
            ->limit(5)
            ->get();

        if ($candidateTransactions->isEmpty()) {
            return; // remain unmatched, to be reviewed later
        }

        // Basic heuristic: pick the closest by date
        $match = $candidateTransactions->first();
        $receipt->update([
            'matched_transaction_id' => $match->id,
            'statement_id' => $match->statement_id,
            'status' => 'matched',
        ]);

        $match->update([
            'matched_receipt_id' => $receipt->id,
        ]);
    }

    public function attemptMatchForStatement(Statement $statement): void
    {
        $unmatchedReceipts = Receipt::query()
            ->whereNull('matched_transaction_id')
            ->whereBetween('receipt_date', [
                Carbon::parse($statement->start_date)->subDays(5),
                Carbon::parse($statement->end_date)->addDays(7),
            ])->get();

        foreach ($unmatchedReceipts as $receipt) {
            $this->attemptMatchForReceipt($receipt);
        }
    }
}


