<?php

namespace App\Services\AI;

use App\Models\Receipt;
use App\Models\ReceiptGroup;
use App\Services\AI\VertexClient;
use Carbon\Carbon;
use Illuminate\Support\Collection;

class ReceiptGroupingService
{
    public function __construct(private VertexClient $vertex) {}

    /**
     * Analyze receipts and suggest groupings based on AI analysis
     */
    public function suggestGroupings(Collection $receipts): array
    {
        if ($receipts->count() < 2) {
            return [];
        }

        $model = config('vertex.models.match', 'gemini-2.5-flash-lite');
        $system = 'You are an expert expense management assistant. Your task is to analyze receipts and identify which ones belong to the same transaction or business expense.

ANALYSIS CRITERIA:
1. MERCHANT SIMILARITY: Receipts from the same business (even with different names)
2. AMOUNT COMPLEMENTARITY: Receipts that together make up a logical total
3. DATE PROXIMITY: Receipts from the same or very close dates
4. BUSINESS CONTEXT: Receipts that logically belong together (e.g., card receipt + itemized receipt)
5. TRANSACTION PATTERNS: Receipts that appear to be parts of the same purchase

COMMON SCENARIOS:
- Card receipt + itemized receipt from same merchant
- Split receipts (e.g., two parts of one receipt due to image splitting)
- Related purchases at same location on same day
- Receipts with complementary amounts that sum to a round number

OUTPUT JSON:
{
  "suggestions": [
    {
      "group_name": "Costco Wholesale Purchase",
      "receipt_ids": [123, 124],
      "confidence": 0.95,
      "reasoning": "Both receipts from Costco on same date, amounts complement each other",
      "grouping_type": "split_receipt"
    }
  ]
}

GROUPING TYPES:
- "split_receipt": Two parts of the same receipt
- "card_and_itemized": Card receipt + detailed itemized receipt
- "related_purchase": Related purchases at same location
- "same_transaction": Same transaction, different receipt types

CONFIDENCE LEVELS:
- 0.9-1.0: Very high confidence (same merchant, complementary amounts, same date)
- 0.7-0.9: High confidence (similar merchant, close amounts, close dates)
- 0.5-0.7: Medium confidence (some similarities, worth reviewing)
- 0.0-0.5: Low confidence (unlikely to be related)';

        $payload = [
            'receipts' => $receipts->map(fn($receipt) => [
                'id' => $receipt->id,
                'date' => $receipt->receipt_date?->format('Y-m-d'),
                'merchant' => $receipt->merchant_name,
                'amount' => $receipt->total_amount,
                'currency' => $receipt->currency,
                'lines_count' => $receipt->lines->count(),
                'has_detailed_lines' => $receipt->lines->where('description', '!=', '')->count() > 0,
            ])->values()->all(),
        ];

        $prompt = json_encode($payload, JSON_UNESCAPED_SLASHES);
        $resp = $this->vertex->generate($model, $prompt, $system, ['responseMimeType' => 'application/json']);
        
        if (!isset($resp['json']) || !is_array($resp['json'])) {
            return [];
        }

        $result = $resp['json'];
        
        if (!isset($result['suggestions']) || !is_array($result['suggestions'])) {
            return [];
        }

        return $result['suggestions'];
    }

    /**
     * Create a receipt group from a suggestion
     */
    public function createGroupFromSuggestion(array $suggestion, int $userId): ReceiptGroup
    {
        $group = ReceiptGroup::create([
            'name' => $suggestion['group_name'] ?? 'Grouped Receipts',
            'description' => $suggestion['reasoning'] ?? 'AI-suggested grouping',
            'grouping_method' => 'ai_suggested',
            'confidence_score' => $suggestion['confidence'] ?? 0.5,
            'grouping_reason' => $suggestion['reasoning'] ?? 'AI analysis suggested these receipts belong together',
            'user_id' => $userId,
        ]);

        // Update receipts to belong to this group
        Receipt::whereIn('id', $suggestion['receipt_ids'] ?? [])
            ->update(['receipt_group_id' => $group->id]);

        return $group;
    }

    /**
     * Analyze a specific receipt against existing groups to find potential matches
     */
    public function findPotentialGroupMatches(Receipt $receipt, Collection $existingGroups): array
    {
        $matches = [];

        foreach ($existingGroups as $group) {
            $confidence = $this->calculateGroupMatchConfidence($receipt, $group);
            
            if ($confidence > 0.5) {
                $matches[] = [
                    'group' => $group,
                    'confidence' => $confidence,
                    'reasoning' => $this->generateMatchReasoning($receipt, $group),
                ];
            }
        }

        // Sort by confidence descending
        usort($matches, fn($a, $b) => $b['confidence'] <=> $a['confidence']);

        return $matches;
    }

    /**
     * Calculate confidence that a receipt belongs to a group
     */
    private function calculateGroupMatchConfidence(Receipt $receipt, ReceiptGroup $group): float
    {
        $confidence = 0.0;

        // Check merchant similarity
        $merchantConfidence = $this->calculateMerchantSimilarity($receipt->merchant_name, $group->primary_merchant);
        $confidence += $merchantConfidence * 0.4;

        // Check date proximity
        $dateConfidence = $this->calculateDateProximity($receipt->receipt_date, $group->date_range);
        $confidence += $dateConfidence * 0.3;

        // Check amount complementarity
        $amountConfidence = $this->calculateAmountComplementarity($receipt->total_amount, $group->total_amount);
        $confidence += $amountConfidence * 0.3;

        return min(1.0, $confidence);
    }

    private function calculateMerchantSimilarity(?string $merchant1, ?string $merchant2): float
    {
        if (!$merchant1 || !$merchant2) {
            return 0.0;
        }

        $m1 = strtolower(trim($merchant1));
        $m2 = strtolower(trim($merchant2));

        if ($m1 === $m2) {
            return 1.0;
        }

        // Check for common variations
        $variations = [
            'costco wholesale' => ['costco', 'costco wholesale corp', 'costco wholesale ltd'],
            'robert todd and son' => ['robert todd & son', 'r todd and son', 'todd and son'],
        ];

        foreach ($variations as $base => $alts) {
            if (in_array($m1, $alts) && in_array($m2, $alts)) {
                return 0.9;
            }
            if ($m1 === $base && in_array($m2, $alts)) {
                return 0.9;
            }
            if ($m2 === $base && in_array($m1, $alts)) {
                return 0.9;
            }
        }

        // Use similarity text
        similar_text($m1, $m2, $percent);
        return $percent / 100;
    }

    private function calculateDateProximity(?string $receiptDate, array $dateRange): float
    {
        if (!$receiptDate || !$dateRange['start']) {
            return 0.0;
        }

        $receipt = Carbon::parse($receiptDate);
        $start = Carbon::parse($dateRange['start']);
        $end = Carbon::parse($dateRange['end']);

        // If receipt date is within the group's date range
        if ($receipt->between($start, $end)) {
            return 1.0;
        }

        // Calculate proximity based on days difference
        $minDays = min($receipt->diffInDays($start), $receipt->diffInDays($end));
        
        if ($minDays <= 1) return 0.9;
        if ($minDays <= 3) return 0.7;
        if ($minDays <= 7) return 0.5;
        if ($minDays <= 14) return 0.3;
        
        return 0.0;
    }

    private function calculateAmountComplementarity(float $receiptAmount, float $groupAmount): float
    {
        $total = $receiptAmount + $groupAmount;
        
        // Check if amounts are complementary (sum to a round number)
        if (abs($total - round($total)) < 0.01) {
            return 0.9;
        }

        // Check if amounts are similar (might be duplicates)
        $ratio = $receiptAmount / $groupAmount;
        if ($ratio > 0.8 && $ratio < 1.2) {
            return 0.7;
        }

        // Check if one is much smaller (might be a partial receipt)
        if ($receiptAmount < $groupAmount * 0.3 || $groupAmount < $receiptAmount * 0.3) {
            return 0.6;
        }

        return 0.0;
    }

    private function generateMatchReasoning(Receipt $receipt, ReceiptGroup $group): string
    {
        $reasons = [];

        if ($this->calculateMerchantSimilarity($receipt->merchant_name, $group->primary_merchant) > 0.7) {
            $reasons[] = "Similar merchant names";
        }

        if ($this->calculateDateProximity($receipt->receipt_date, $group->date_range) > 0.7) {
            $reasons[] = "Same or very close dates";
        }

        if ($this->calculateAmountComplementarity($receipt->total_amount, $group->total_amount) > 0.7) {
            $reasons[] = "Complementary amounts";
        }

        return implode(', ', $reasons) ?: 'Some similarities detected';
    }
}
