<?php

namespace App\Http\Livewire\Imports;

use Gemini;
use OpenAI;
use App\Models\Price;
use App\Models\Sizes;
use Gemini\Data\Blob;
use League\Csv\Writer;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Helper\Functions;
use Spatie\PdfToText\Pdf;
use App\Models\Colourways;
use Gemini\Enums\MimeType;
use App\Models\Departments;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use Smalot\PdfParser\Parser;
use App\Models\StyleVersions;
use Livewire\WithFileUploads;
use App\Models\CustomerOrders;
use LucianoTonet\GroqPHP\Groq;
use App\Models\CustomerAddress;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use App\Models\CustomerOrderLines;
use Illuminate\Support\Facades\DB;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Gate;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\ImportExcelWithValues;
use Spatie\PdfToImage\Pdf as PdfToImage;
use App\Models\CustomerOrderLineQuantities;


class AiOrdersKwa extends BaseComponent
{
    use WithFileUploads;

    public $orderFiles = [];
    public $answer;
    public $question = '';
    public $disableUploadButton = false;
    public $season;
    public $files;
    public $customer=231;

	#[Computed]
	public function customers(){
		return Customer::allCached();
	}
    #[Computed]
	public function seasons(){
		return Seasons::allCached();
	}
    #[Computed]
	public function departments(){
		return Departments::allCached();
	}

    #[Computed]
	public function addresses(){
		return CustomerAddress::where('customer_id', $this->customer)->get();
	}

    #[Computed]
    public function styleVersions(){
        return StyleVersions::with('styles')->whereRelation('styles', 'seasons_id', $this->season)->whereRelation('styles', 'customers_id', $this->customer)->get();
    }
    public function colourways($style_version_id){
		return Colourways::with('style_versions.styles.designs')->where('style_versions_id', $style_version_id)->get();
	}

    public function updated($var, $val){
        if(str_contains($var, "style_version_id")){
            $varList = explode('.', $var);
            $file = $varList[1];
            $po = $varList[3];
            $style = $varList[5];

            $this->files[$file]['purchase_orders'][$po]['styles'][$style]['model'] = $this->styleVersions->where('id', $val)->first();
            foreach($this->files[$file]['purchase_orders'][$po]['styles'][$style]['colorways'] as $c=>$colourway){
                $this->files[$file]['purchase_orders'][$po]['styles'][$style]['colorways'][$c]['model'] = null;
                $this->files[$file]['purchase_orders'][$po]['styles'][$style]['colorways'][$c]['colorway_id'] = null;
            }
        }
        if(str_contains($var, "colorways")){
            $varList = explode('.', $var);
            $file = $varList[1];
            $po = $varList[3];
            $style = $varList[5];
            $cw = $varList[7];

            $this->files[$file]['purchase_orders'][$po]['styles'][$style]['colorways'][$cw]['model'] = Colourways::find($val);
        }
    }

    // Validation rules
    #[Validate([
        'orderFiles.*' => ['required', 'file', 'mimes:csv,xls,xlsx,pdf,jpg,jpeg,png'], // Validate uploaded files

        // Basic order validation
        'files.*.purchase_orders.*.po_number' => ['required', 'string', 'max:50'], // PO number must be a string
        'files.*.purchase_orders.*.date' => ['required', 'date'], // Order date must be valid
        'files.*.purchase_orders.*.incoterms' => ['nullable', 'string', 'max:20'], // Optional incoterms
        'files.*.purchase_orders.*.delivery_address_id' => ['nullable', 'integer', 'exists:customer_addresses,id'], // Address ID must exist
        'files.*.purchase_orders.*.delivery_address.line1' => ['nullable', 'string', 'max:255'], // Optional delivery address
        'files.*.purchase_orders.*.delivery_address.line2' => ['nullable', 'string', 'max:255'],
        'files.*.purchase_orders.*.delivery_address.line3' => ['nullable', 'string', 'max:255'],
        'files.*.purchase_orders.*.delivery_address.city' => ['nullable', 'string', 'max:100'],
        'files.*.purchase_orders.*.delivery_address.county' => ['nullable', 'string', 'max:100'],
        'files.*.purchase_orders.*.delivery_address.postcode' => ['nullable', 'string', 'max:20'],
        'files.*.purchase_orders.*.delivery_address.country' => ['nullable', 'string', 'max:100'], // Must match a valid country

        // Colorway-level validation
        'files.*.purchase_orders.*.styles.*.colorways.*.customer_exfty' => ['required', 'date'], // Customer ex-factory date
        'files.*.purchase_orders.*.styles.*.colorways.*.customer_into_wh' => ['required', 'date'], // Customer into warehouse date
        'files.*.purchase_orders.*.styles.*.colorways.*.colorway_id' => ['nullable', 'integer', 'exists:colourways,id'], // Colorway ID must exist

        // Size quantities validation
        'files.*.purchase_orders.*.styles.*.colorways.*.size_quantities.*.size' => ['required', 'string', 'max:20'], // Size is required
        'files.*.purchase_orders.*.styles.*.colorways.*.size_quantities.*.quantity' => ['required', 'integer', 'min:1'], // Quantity must be a positive integer
        // 'files.*.purchase_orders.*.styles.*.colorways.*.size_quantities.*.sku' => ['nullable', 'string', 'max:50'], // Optional SKU
        // 'files.*.purchase_orders.*.styles.*.colorways.*.size_quantities.*.barcode' => ['nullable', 'string', 'max:50'], // Optional barcode
        // 'files.*.purchase_orders.*.styles.*.colorways.*.size_quantities.*.price' => ['nullable', 'numeric', 'min:0'], // Optional price must be non-negative
        // 'files.*.purchase_orders.*.styles.*.colorways.*.size_quantities.*.quote' => ['nullable', 'numeric', 'min:0'], // Optional quote must be non-negative
    ])]

    public function uploadOrders()
    {
        if (empty($this->orderFiles)) {
            session()->flash('message', 'No files uploaded.');
            return;
        }
        $this->files = [];

        // Loop through each uploaded file
        foreach ($this->orderFiles as $file) {
            $file = base64_encode(file_get_contents($file->getRealPath()));
            if($file){
                $this->files[] = $this->sendToAI($file);
            }
            else{
                dd("file empty");
            }
        }
    }

    private function sendToAI($poText)
    {
        $yourApiKey = getenv('GEMINI_API_KEY');
        $client = Gemini::client($yourApiKey);

        $count = 0;
        $data = [];
        while(!$data && $count < 3){

            $result = $client
                ->geminiFlash()
                ->generateContent([
                    '
                        I will send JSON of colourways and styles in our database. Then I will send customer POs.
                        Match the items in the POs to the items in the database.
                        Double check all data and calculations to make sure youve included line on the POs and ensured the data for each line is in the right place. E.g. Make sure the qty for size S refers to size S and arent being crossed.
                        Make sure that the total qty on the PO adds up to the total of the qtys listed in the JSON.
                        If no incoterm is mentioned, leave it blank.
                        If phase is mentioned, use that for phase_id, otherwise us 1 for Jan-Jun and 2 for Jul-Dec.
                        There will also be a list of customer addresses listed. If the address roughly matches (e.g. same postcode/country), include the address_id, if not then leave the delivery_address_id blank.
                        Do not use the billing addresss as the  - just leave it blank if there is no address to ship to.
                        If using prices to match to the right version, prioritise prices that are confirmed. In JSON use `price` as the customer purchase order price and `quote` as the value in the database. Use the price before discount - pre discount unit cost.
                        Ensure the JSON includes all fields from the example below.
                        If you cant accurately match anything, return an error description as a string.
                        Ensure that the colorway_id, style_version_id and designs_id is filled with the correct value from the database.
                        The barcode may be blank - do not just use the SKU instead, leave it blank.
                        Do not add extra fields into the JSON. Ensure the date and other datatypes are in the formats in the example.
                        Provide the ENTIRE output regardless of length. Every PO should be on there encapsulated by the purchase_orders in json.
                        ENSURE YOU HAVE EVERY SIZE LISTED ON THE PO.
                        e.g.
                        {
                            "purchase_orders": [
                                {
                                    "po_number": "123456789", //ensure this read directly from the customer PO and not the database export
                                    "phase_id": "2",
                                    "date": "2023-09-01",
                                    "incoterms": "EXW",
                                    "delivery_address_id": 1,
                                    "styles": [
                                        {
                                            "designs_id": "1235",
                                            "style_version_id": 6023,
                                            "colorways": [
                                                {
                                                    "colorway_id": 4
                                                    "customer_exfty": "2023-09-15",
                                                    "customer_into_wh": "2023-09-25",
                                                    "size_quantities": [
                                                        { "size": "S", "quantity": 10, "sku": "123456", "barcode": "123456", "price": 10.99, "quote": 13.55 },
                                                        { "size": "M", "quantity": 15, "sku": "123456", "barcode": "123456", "price": 10.99, "quote": 13.55 },
                                                        { "size": "L", "quantity": 12, "sku": "123456", "barcode": "123456", "price": 10.99, "quote": 13.55 }
                                                    ],
                                                }
                                            ]
                                        }
                                    ]
                                },
                            ],
                        }
                    '
                    . json_encode($this->getData()) . json_encode($this->getCustomerAddresses())
                    ,
                    new Blob(
                        mimeType: MimeType::APPLICATION_PDF,
                        data: $poText
                    )
                ]);

            $data = json_decode(str_replace(['```json', '```'], '', $result->text()), 1);

            // dd($data);

            if(isset($data['purchase_orders'])){
                //CHECK IF SV AND C EXIST
                foreach($data['purchase_orders'] as $o=>$order){
                    foreach($order['styles'] as $s=>$style){
                        $styleVersionDB = StyleVersions::with('styles.designs')->find($style['style_version_id']);
                        if($styleVersionDB){
                            $data['purchase_orders'][$o]['styles'][$s]['model'] = $styleVersionDB;

                            foreach($style['colorways'] as $c=>$colorway){
                                if(array_key_exists('colorway_id', $colorway)){
                                    $colourwayDB = Colourways::find($colorway['colorway_id']);
                                    if($colourwayDB){
                                        $data['purchase_orders'][$o]['styles'][$s]['colorways'][$c]['model'] = $colourwayDB;
                                    }
                                    else{
                                        session()->flash('message', 'Error parsing document. Colourway not matched.');
                                    }
                                }
                                else{
                                    dd($data['purchase_orders']);
                                }
                            }
                        }
                        else{
                            session()->flash('message', 'Error parsing document. Style Version not matched.');
                        }
                    }
                }
            }

            $count++;
        }


        if(!isset($data['purchase_orders'])){
            dd($data);
        }

        return $data;
    }

    #[On('render')]
    public function render()
    {
        // Gate::authorize('admin');
        if($this->buttonDisabled()){
            session()->flash('message', 'Max 10 files.');
            session()->flash('alert-class', 'alert-warning');
        }
        return view('livewire.imports.ai-orders-kwa');
    }

    private function buttonDisabled(){
        return count($this->orderFiles) > 10;
    }

    public function getData(){
        $results = DB::select("
        SELECT
            customer_ref,
            d.description AS 'style_description',
            c.name AS 'colourway',
            designs_id,
            styles_id,
            style_versions_id AS style_version_id,
            c.id AS colorway_id,
            seasons.description AS 'season',
            (SELECT
                JSON_ARRAYAGG(
                    JSON_OBJECT(
                        'id', p.id,
                        'phase_id', p.phase_id,
                        'quote', p.quote,
                        'cmt', p.cmt,
                        'quote_status', p.quote_status,
                        'created_at', p.created_at
                    )
                )
            FROM
                prices p
            WHERE
                p.style_versions_id = sv.id
            ) AS prices
        FROM
            colourways c
                JOIN
            style_versions sv ON sv.id = c.style_versions_id
                AND sv.deleted_at IS NULL
                JOIN
            styles s ON s.id = sv.styles_id
                AND s.deleted_at IS NULL
                AND s.customers_id = $this->customer
                AND s.seasons_id = $this->season
                AND s.cancelled = 0
                AND s.departments_id = 2
                JOIN
            designs d ON d.id = s.designs_id
                AND d.deleted_at IS NULL
                JOIN
            gauges g ON g.id = sv.gauge_id
                JOIN
            seasons ON seasons.id = s.seasons_id
        WHERE
            c.deleted_at IS NULL
		AND
			c.cancelled = 0
        ");

        return json_decode(json_encode($results), TRUE);
    }
    public function getCustomerAddresses(){
        $results = DB::select("
        SELECT
            *
        FROM
            customer_addresses
        WHERE
            deleted_at IS NULL
        AND
            customer_id = 6
        ");

        return json_decode(json_encode($results), TRUE);
    }

    public function createOrders()
    {
        $this->validate();

        // Initialize message collections
        $errorMessages = [];
        $savedOrders = [];

        DB::beginTransaction();

        try {
            foreach ($this->files as $file) {
                foreach ($file['purchase_orders'] as $o => $order) {
                    // Validate the purchase order structure
                    if (!$this->isValidOrder($order)) {
                        $errorMessages[] = "Invalid purchase order structure for PO number: {$order['po_number']}";
                        \Log::error("Invalid purchase order structure", ['order' => $order]);
                        continue;
                    }

                    if (CustomerOrders::where('customer_po', $order['po_number'])->exists()) {
                        $errorMessages[] = "PO already exists: {$order['po_number']}";
                        \Log::error("PO already exists: {$order['po_number']}");
                        continue;
                    }

                    // Create the customer order
                    $newOrder = new CustomerOrders;
                    $newOrder->customer_po = $order['po_number'];
                    $newOrder->order_date = $order['date'];
                    $newOrder->incoterms = $order['incoterms'] ?? null;
                    $newOrder->departments_id = 2;
                    $newOrder->customers_id = $this->customer;
                    $newOrder->seasons_id = $this->season;
                    $newOrder->phase_id = $order['phase_id'] ?? null;

                    if (!empty($order['delivery_address_id'])) {
                        $newOrder->customer_addresses_id = $order['delivery_address_id'];
                    } else {
                        // Validate delivery address structure
                        if (!$this->isValidAddress($order['delivery_address'])) {
                            $errorMessages[] = "Invalid delivery address structure for PO number: {$order['po_number']}";
                            \Log::error("Invalid delivery address structure", ['address' => $order['delivery_address']]);
                            continue;
                        }

                        $country = Countries::where('country', $order['delivery_address']['country'])->first();
                        if (!$country) {
                            $errorMessages[] = "Invalid country for address in PO number: {$order['po_number']}";
                            \Log::error("Invalid country", ['country' => $order['delivery_address']['country']]);
                            continue;
                        }

                        $address = CustomerAddress::firstOrCreate(
                            [
                                'line1' => $order['delivery_address']['line1'],
                                'postcode' => $order['delivery_address']['postcode'],
                            ],
                            [
                                'line2' => $order['delivery_address']['line2'] ?? '',
                                'line3' => $order['delivery_address']['line3'] ?? '',
                                'city' => $order['delivery_address']['city'] ?? '',
                                'county' => $order['delivery_address']['county'] ?? '',
                                'postcode' => $order['delivery_address']['postcode'] ?? '',
                                'country' => $country->id,
                            ]
                        );
                        $newOrder->customer_addresses_id = $address->id;
                    }

                    $newOrder->save();
                    $savedOrders[] = $order['po_number']; // Collect saved order POs

                    foreach ($order['styles'] as $style) {
                        foreach ($style['colorways'] as $colorway) {
                            $newLine = new CustomerOrderLines;
                            $newLine->customer_orders_id = $newOrder->id;
                            $newLine->colourways_id = $colorway['colorway_id'] ?? null;
                            $newLine->factory_cust_date = $colorway['customer_exfty'] ?? null;
                            $newLine->wh_cust_date = $colorway['customer_into_wh'] ?? null;
                            $newLine->start_knit = $colorway['start_date'] ?? null;
                            $newLine->save();

                            foreach ($colorway['size_quantities'] as $size) {
                                if (!$this->isValidSize($size)) {
                                    $errorMessages[] = "Invalid size structure for PO number: {$order['po_number']}";
                                    \Log::error("Invalid size structure", ['size' => $size]);
                                    continue;
                                }

                                $sz = Sizes::firstOrCreate(['name' => $size['size']]);
                                $newSize = new CustomerOrderLineQuantities;
                                $newSize->customer_order_lines_id = $newLine->id;
                                $newSize->sizes_id = $sz->id;
                                $newSize->qty = $size['quantity'] ?? 0;
                                $newSize->SKU = $size['sku'] ?? '';
                                $newSize->barcode = $size['barcode'] ?? '';

                                                                                        // Price resolution cache is automatically handled
                                                                                        // No need to set prices_id as it's no longer used for caching


                                // dd($newSize);
                                $newSize->save();

                                if (!isset($size['quote'])) {
                                    Price::updateOrCreate(
                                        [
                                            'style_versions_id' => $colorway['model']['style_versions_id'],
                                        ],
                                        [
                                            'cmt' => $size['cmt'] ?? 0,
                                            'quote' => $size['price'],
                                            'quote_status' => 'confirmed',
                                        ]
                                    );
                                }
                            }

                            $newShipmentLine = new ShipmentLine;
                            $newShipmentLine->customer_order_lines_id = $newLine->id;
                            $newShipmentLine->exfty = $colorway['customer_exfty'] ?? null;
                            $newShipmentLine->save();

                            foreach ($colorway['size_quantities'] as $size) {
                                if (!$this->isValidSize($size)) {
                                    $errorMessages[] = "Invalid size structure for PO number: {$order['po_number']}";
                                    \Log::error("Invalid size structure", ['size' => $size]);
                                    continue;
                                }

                                $sz = Sizes::firstOrCreate(['name' => $size['size']]);
                                $newSize = new ShipmentLineSizes;
                                $newSize->shipment_line_id = $newShipmentLine->id;
                                $newSize->sizes_id = $sz->id;
                                $newSize->qty = $size['quantity'] ?? 0;
                                $newSize->save();
                            }
                        }
                    }
                }
            }

            DB::commit(); // Commit transaction if no exceptions occur


            // Flash success and error messages
            session()->flash('messages', [
                'errors' => $errorMessages,
                'saved' => $savedOrders,
            ]);

            $this->reset();
        } catch (\Exception $e) {
            DB::rollBack();                                     // Roll back on error
            \Log::error('Error creating orders', [
                'error' => $e->getMessage()
            ]);

            // 1. Always show your friendly text + any domain-level error messages
            $errors = array_merge(
                $errorMessages,
                ['An unexpected error occurred while processing orders.']
            );

            // 2. Only reveal the technical message when APP_DEBUG=true
            if (config('app.debug')) {
                $errors[] = $e->getMessage();
            }

            // 3. Flash to the session
            session()->flash('messages', [
                'errors' => $errors,
            ]);
        }
    }


    private function isValidOrder($order)
    {
        return isset($order['po_number'], $order['date'], $order['styles']);
    }

    private function isValidAddress($address)
    {
        return isset($address['line1'], $address['postcode'], $address['country']);
    }

    private function isValidSize($size)
    {
        return isset($size['size'], $size['quantity']);
    }

}
