<?php

namespace App\Livewire\Admin;

use App\Models\Item;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Livewire\WithFileUploads;

class ColourwaysPdfImport extends Component
{
    use WithFileUploads;

    public Item $item;

    #[Validate('required|file|mimes:pdf|max:51200')]
    public $pdf;

    public function mount(Item $item): void
    {
        $this->item = $item;
        logger()->info('ColourwaysPdfImport component mounted', ['item_id' => $item->id]);
    }

    public function testConnection(): void
    {
        logger()->info('Test connection method called');
        $this->dispatch('notify', message: 'Component is working!');
    }
    
    public function checkFileState(): void
    {
        logger()->info('File state check', [
            'has_pdf' => isset($this->pdf),
            'pdf_value' => $this->pdf,
            'pdf_type' => gettype($this->pdf),
            'pdf_class' => $this->pdf ? get_class($this->pdf) : 'null'
        ]);
        
        if ($this->pdf) {
            logger()->info('PDF details', [
                'original_name' => $this->pdf->getClientOriginalName(),
                'size' => $this->pdf->getSize(),
                'mime_type' => $this->pdf->getMimeType(),
                'error' => $this->pdf->getError()
            ]);
        }
        
        $this->dispatch('notify', message: 'File state logged to console');
    }

    public function import(): void
    {
        try {
            logger()->info('PDF import started', ['item_id' => $this->item->id]);
            
            // Check if we have a file
            if (!$this->pdf) {
                logger()->error('No PDF file in component');
                $this->addError('pdf', 'No file received by server.');
                return;
            }
            
            logger()->info('PDF file received', [
                'original_name' => $this->pdf->getClientOriginalName(),
                'size' => $this->pdf->getSize(),
                'mime_type' => $this->pdf->getMimeType(),
                'error' => $this->pdf->getError()
            ]);
            
            $this->validate();
            logger()->info('Validation passed');

            // Ensure temp dir exists/writable
            Storage::disk('local')->makeDirectory('temp');
            logger()->info('Temp directory ensured');

            if (!$this->pdf) {
                logger()->error('No PDF file received');
                $this->addError('pdf', 'No file received by server.');
                return;
            }

            logger()->info('PDF file details', [
                'original_name' => $this->pdf->getClientOriginalName(),
                'size' => $this->pdf->getSize(),
                'mime_type' => $this->pdf->getMimeType()
            ]);

            $path = $this->pdf->store('temp', 'local');
            logger()->info('PDF stored at: ' . $path);
            
            // Get the full path to the stored PDF file
            $fullPdfPath = Storage::disk('local')->path($path);
            logger()->info('Full PDF path: ' . $fullPdfPath);
            
            $tabulaJar = base_path('tabula/tabula-1.0.5-jar-with-dependencies.jar');
            if (!file_exists($tabulaJar)) {
                $this->dispatch('notify', message: 'Tabula jar missing at tabula/tabula-1.0.5-jar-with-dependencies.jar');
                return;
            }

            $outputJson = storage_path('app/'.uniqid('tabula_').'.json');
            $cmd = sprintf('java -jar %s -p all -f JSON -o %s %s', escapeshellarg($tabulaJar), escapeshellarg($outputJson), escapeshellarg($fullPdfPath));
            logger()->info('Tabula command: ' . $cmd);
            $code = null; $out = [];
            exec($cmd.' 2>&1', $out, $code);
            
            logger()->info('Tabula execution result', [
                'exit_code' => $code,
                'output' => $out,
                'output_file_exists' => file_exists($outputJson),
                'output_file_path' => $outputJson
            ]);

            if ($code !== 0 || !file_exists($outputJson)) {
                $this->addError('pdf', 'Failed to parse PDF: '.implode("\n", $out));
                return;
            }

            $json = json_decode(file_get_contents($outputJson), true);
            
            // Clean up temporary files
            Storage::disk('local')->delete($path);
            unlink($outputJson);
            
            // Clear the file input
            $this->pdf = null;
            
            // Extract colour codes from the parsed data
            $colourCodes = $this->extractColourCodes($json);
            
            // Dump the extracted colour codes to the browser
            dd('PDF Parsed Successfully!', [
                'item_id' => $this->item->id,
                'original_filename' => $this->pdf ? $this->pdf->getClientOriginalName() : 'Unknown',
                'extracted_colour_codes' => $colourCodes,
                'total_colours_found' => count($colourCodes),
                'raw_parsed_data' => $json
            ]);
            
        } catch (\Throwable $e) {
            report($e);
            $this->addError('pdf', 'Upload failed: '.$e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.admin.colourways-pdf-import');
    }
}


