<?php

namespace App\Livewire\Admin;

use App\Models\Item;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Livewire\WithFileUploads;

class ColourwaysPdfImport extends Component
{
    use WithFileUploads;

    public Item $item;

    #[Validate('required|file|mimes:pdf|max:51200')]
    public $pdf;

    public function mount(Item $item): void
    {
        $this->item = $item;
    }

    public function import(): void
    {
        try {
            $this->validate();

            // Ensure temp dir exists/writable
            Storage::disk('local')->makeDirectory('temp');

            if (!$this->pdf) {
                $this->addError('pdf', 'No file received by server.');
                return;
            }

            $path = $this->pdf->store('temp', 'local');
            
            $tabulaJar = base_path('tabula/tabula-1.0.5-jar-with-dependencies.jar');
            if (!file_exists($tabulaJar)) {
                $this->dispatch('notify', message: 'Tabula jar missing at tabula/tabula-1.0.5-jar-with-dependencies.jar');
                return;
            }

            $outputJson = storage_path('app/'.uniqid('tabula_').'.json');
            $cmd = sprintf('java -jar %s -f JSON -o %s %s', escapeshellarg($tabulaJar), escapeshellarg($outputJson), escapeshellarg(storage_path('app/'.$path)));
            $code = null; $out = [];
            exec($cmd.' 2>&1', $out, $code);

            if ($code !== 0 || !file_exists($outputJson)) {
                $this->addError('pdf', 'Failed to parse PDF: '.implode("\n", $out));
                return;
            }

            $json = json_decode(file_get_contents($outputJson), true);
            // For now, dump to the browser console (avoids huge HTML output)
            logger()->info('tabula_output', ['item_id' => $this->item->id, 'data' => $json]);
            $this->dispatch('notify', message: 'PDF parsed. Check storage/logs/laravel.log.');
            
        } catch (\Throwable $e) {
            report($e);
            $this->addError('pdf', 'Upload failed: '.$e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.admin.colourways-pdf-import');
    }
}


