<?php

namespace App\Livewire\Admin;

use App\Models\Item;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Livewire\WithFileUploads;

class ColourwaysPdfImport extends Component
{
    use WithFileUploads;

    public Item $item;

    #[Validate('required|file|mimes:pdf|max:51200')]
    public $pdf;

    public function mount(Item $item): void
    {
        $this->item = $item;
        logger()->info('ColourwaysPdfImport component mounted', ['item_id' => $item->id]);
    }

    public function testConnection(): void
    {
        logger()->info('Test connection method called');
        $this->dispatch('notify', message: 'Component is working!');
    }
    
    public function checkFileState(): void
    {
        logger()->info('File state check', [
            'has_pdf' => isset($this->pdf),
            'pdf_value' => $this->pdf,
            'pdf_type' => gettype($this->pdf),
            'pdf_class' => $this->pdf ? get_class($this->pdf) : 'null'
        ]);
        
        if ($this->pdf) {
            logger()->info('PDF details', [
                'original_name' => $this->pdf->getClientOriginalName(),
                'size' => $this->pdf->getSize(),
                'mime_type' => $this->pdf->getMimeType(),
                'error' => $this->pdf->getError()
            ]);
        }
        
        $this->dispatch('notify', message: 'File state logged to console');
    }

    public function import(): void
    {
        try {
            logger()->info('PDF import started', ['item_id' => $this->item->id]);
            
            // Check if we have a file
            if (!$this->pdf) {
                logger()->error('No PDF file in component');
                $this->addError('pdf', 'No file received by server.');
                return;
            }
            
            logger()->info('PDF file received', [
                'original_name' => $this->pdf->getClientOriginalName(),
                'size' => $this->pdf->getSize(),
                'mime_type' => $this->pdf->getMimeType(),
                'error' => $this->pdf->getError()
            ]);
            
            $this->validate();
            logger()->info('Validation passed');

            // Ensure temp dir exists/writable
            Storage::disk('local')->makeDirectory('temp');
            logger()->info('Temp directory ensured');

            if (!$this->pdf) {
                logger()->error('No PDF file received');
                $this->addError('pdf', 'No file received by server.');
                return;
            }

            logger()->info('PDF file details', [
                'original_name' => $this->pdf->getClientOriginalName(),
                'size' => $this->pdf->getSize(),
                'mime_type' => $this->pdf->getMimeType()
            ]);

            $path = $this->pdf->store('temp', 'local');
            logger()->info('PDF stored at: ' . $path);
            
            // Get the full path to the stored PDF file
            $fullPdfPath = Storage::disk('local')->path($path);
            logger()->info('Full PDF path: ' . $fullPdfPath);
            
            $tabulaJar = base_path('tabula/tabula-1.0.5-jar-with-dependencies.jar');
            if (!file_exists($tabulaJar)) {
                $this->dispatch('notify', message: 'Tabula jar missing at tabula/tabula-1.0.5-jar-with-dependencies.jar');
                return;
            }

            $outputJson = storage_path('app/'.uniqid('tabula_').'.json');
            $cmd = sprintf('java -jar %s -p all -f JSON -o %s %s', escapeshellarg($tabulaJar), escapeshellarg($outputJson), escapeshellarg($fullPdfPath));
            logger()->info('Tabula command: ' . $cmd);
            $code = null; $out = [];
            exec($cmd.' 2>&1', $out, $code);
            
            logger()->info('Tabula execution result', [
                'exit_code' => $code,
                'output' => $out,
                'output_file_exists' => file_exists($outputJson),
                'output_file_path' => $outputJson
            ]);

            if ($code !== 0 || !file_exists($outputJson)) {
                $this->addError('pdf', 'Failed to parse PDF: '.implode("\n", $out));
                return;
            }

            $json = json_decode(file_get_contents($outputJson), true);
            
            // Clean up temporary files
            Storage::disk('local')->delete($path);
            unlink($outputJson);
            
            // Clear the file input
            $this->pdf = null;
            
            // Extract colour codes from the parsed data
            $result = $this->extractColourCodes($json);
            
            // Dump the extracted colour codes to the browser
            dd('PDF Parsed Successfully!', [
                'item_id' => $this->item->id,
                'original_filename' => $this->pdf ? $this->pdf->getClientOriginalName() : 'Unknown',
                'extracted_colour_codes' => $result['codes'],
                'total_colours_found' => count($result['codes']),
                'colour_codes_with_positions' => $result['with_positions'],
                'raw_parsed_data' => $json
            ]);
            
        } catch (\Throwable $e) {
            report($e);
            $this->addError('pdf', 'Upload failed: '.$e->getMessage());
        }
    }

    /**
     * Extract colour codes from Tabula JSON output
     * Looks for 6-digit numbers that represent colour codes
     * Sorts by grid coordinates (A1, A2, B1, B2, etc.)
     */
    private function extractColourCodes(array $jsonData): array
    {
        $colourCodesWithGrid = [];
        
        // Extract text with grid coordinate information
        $this->extractTextWithGrid($jsonData, $colourCodesWithGrid);
        
        // Sort by grid coordinates (A1, A2, A3, B1, B2, B3, etc.)
        usort($colourCodesWithGrid, function($a, $b) {
            // First sort by column (A, B, C, etc.)
            if ($a['column'] !== $b['column']) {
                return strcmp($a['column'], $b['column']);
            }
            // Then sort by row (1, 2, 3, etc.)
            return $a['row'] <=> $b['row'];
        });
        
        // Extract just the colour codes in order
        $colourCodes = array_column($colourCodesWithGrid, 'code');
        
        logger()->info('Colour codes extracted with grid coordinates', [
            'total_found' => count($colourCodes),
            'codes' => $colourCodes,
            'with_grid' => $colourCodesWithGrid
        ]);
        
        return [
            'codes' => $colourCodes,
            'with_positions' => $colourCodesWithGrid
        ];
    }
    
    /**
     * Recursively extract text with grid coordinate information
     */
    private function extractTextWithGrid($data, &$colourCodesWithGrid): void
    {
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                if ($key === 'text' && is_string($value)) {
                    // Find 6-digit colour codes in this text, excluding ID numbers and other non-colour codes
                    preg_match_all('/\b(?!4S-|ID\s*Nr\.|Nr\.)\d{6}\b/', $value, $matches, PREG_OFFSET_CAPTURE);
                    
                    if (!empty($matches[0])) {
                        foreach ($matches[0] as $match) {
                            // Look for grid coordinates (A1, B2, C3, etc.) in the surrounding text
                            $gridCoords = $this->findGridCoordinates($data, $value);
                            
                            $colourCodesWithGrid[] = [
                                'code' => $match[0],
                                'text' => $value,
                                'column' => $gridCoords['column'],
                                'row' => $gridCoords['row'],
                                'grid_ref' => $gridCoords['grid_ref'],
                                'raw_data' => $data // Store raw data for debugging
                            ];
                        }
                    }
                } else {
                    $this->extractTextWithGrid($value, $colourCodesWithGrid);
                }
            }
        }
    }
    
    /**
     * Find grid coordinates in the text or surrounding context
     */
    private function findGridCoordinates($data, $text): array
    {
        // Look for grid coordinates in the current text
        if (preg_match('/\b([A-Z])(\d+)\b/', $text, $matches)) {
            return [
                'column' => $matches[1],
                'row' => (int) $matches[2],
                'grid_ref' => $matches[0]
            ];
        }
        
        // Look for grid coordinates in surrounding text blocks
        $surroundingText = $this->getSurroundingText($data);
        if (preg_match('/\b([A-Z])(\d+)\b/', $surroundingText, $matches)) {
            return [
                'column' => $matches[1],
                'row' => (int) $matches[2],
                'grid_ref' => $matches[0]
            ];
        }
        
        // Default fallback
        return [
            'column' => 'Z',
            'row' => 999,
            'grid_ref' => 'Z999'
        ];
    }
    
    /**
     * Get surrounding text context to find grid coordinates
     */
    private function getSurroundingText($data): string
    {
        $text = '';
        
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                if ($key === 'text' && is_string($value)) {
                    $text .= ' ' . $value;
                } elseif (is_array($value)) {
                    $text .= ' ' . $this->getSurroundingText($value);
                }
            }
        }
        
        return trim($text);
    }

    public function render()
    {
        return view('livewire.admin.colourways-pdf-import');
    }
}


