<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Services\LibraryService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class LibraryController extends Controller
{
    public function __construct(
        private LibraryService $libraryService
    ) {}

    public function categories(): JsonResponse
    {
        $categories = $this->libraryService->getCategories();
        
        return response()->json([
            'data' => $categories->map(fn($category) => [
                'id' => $category->id,
                'name' => $category->name,
                'slug' => $category->slug,
                'description' => $category->description,
                'icon' => $category->icon,
                'color' => $category->color,
                'trackCount' => $category->tracks->count(),
            ]),
        ]);
    }

    public function tracks(Request $request): JsonResponse
    {
        $filters = $request->only(['category_id', 'voice_type', 'difficulty', 'search']);
        $tracks = $this->libraryService->getTracks($filters);
        
        return response()->json([
            'data' => $tracks->items(),
            'meta' => [
                'current_page' => $tracks->currentPage(),
                'last_page' => $tracks->lastPage(),
                'per_page' => $tracks->perPage(),
                'total' => $tracks->total(),
            ],
        ]);
    }

    public function track(string $slug): JsonResponse
    {
        $track = $this->libraryService->getTrack($slug);
        
        if (!$track) {
            return response()->json([
                'error' => 'Track not found',
                'message' => 'The requested track could not be found.',
            ], Response::HTTP_NOT_FOUND);
        }
        
        return response()->json([
            'data' => [
                'id' => $track->id,
                'title' => $track->title,
                'slug' => $track->slug,
                'description' => $track->description,
                'transcript' => $track->transcript,
                'audioUrl' => $this->libraryService->getTrackAudioUrl($track),
                'duration' => $track->duration,
                'tags' => $track->tags,
                'voiceType' => $track->voice_type,
                'difficulty' => $track->difficulty,
                'playCount' => $track->play_count,
                'category' => [
                    'id' => $track->category->id,
                    'name' => $track->category->name,
                    'slug' => $track->category->slug,
                    'color' => $track->category->color,
                ],
                'createdAt' => $track->created_at,
                'updatedAt' => $track->updated_at,
            ],
        ]);
    }

    public function popular(): JsonResponse
    {
        $tracks = $this->libraryService->getPopularTracks();
        
        return response()->json([
            'data' => $tracks->map(fn($track) => [
                'id' => $track->id,
                'title' => $track->title,
                'slug' => $track->slug,
                'description' => $track->description,
                'duration' => $track->duration,
                'playCount' => $track->play_count,
                'category' => [
                    'name' => $track->category->name,
                    'color' => $track->category->color,
                ],
            ]),
        ]);
    }
}
