<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class LibraryController extends Controller
{
    public function index(): JsonResponse
    {
        $categories = LibraryCategory::active()
            ->ordered()
            ->with(['tracks' => function ($query) {
                $query->active()->orderBy('title');
            }])
            ->get();

        return response()->json([
            'categories' => $categories->map(function ($category) {
                return [
                    'id' => $category->id,
                    'name' => $category->name,
                    'slug' => $category->slug,
                    'description' => $category->description,
                    'icon' => $category->icon,
                    'color' => $category->color,
                    'tracksCount' => $category->tracks->count(),
                    'tracks' => $category->tracks->map(function ($track) {
                        return [
                            'id' => $track->id,
                            'title' => $track->title,
                            'slug' => $track->slug,
                            'description' => $track->description,
                            'duration' => $track->duration,
                            'formattedDuration' => $track->formatted_duration,
                            'voiceType' => $track->voice_type,
                            'difficultyLevel' => $track->difficulty_level,
                            'tags' => $track->tags,
                            'playCount' => $track->play_count,
                            'rating' => $track->rating,
                        ];
                    }),
                ];
            }),
        ]);
    }

    public function categories(): JsonResponse
    {
        $categories = LibraryCategory::active()
            ->ordered()
            ->withCount('tracks')
            ->get();

        return response()->json([
            'categories' => $categories->map(function ($category) {
                return [
                    'id' => $category->id,
                    'name' => $category->name,
                    'slug' => $category->slug,
                    'description' => $category->description,
                    'icon' => $category->icon,
                    'color' => $category->color,
                    'tracksCount' => $category->tracks_count,
                ];
            }),
        ]);
    }

    public function tracks(Request $request): JsonResponse
    {
        $query = LibraryTrack::active()
            ->with('category')
            ->orderBy('title');

        if ($request->has('category')) {
            $query->whereHas('category', function ($q) use ($request) {
                $q->where('slug', $request->category);
            });
        }

        if ($request->has('difficulty')) {
            $query->byDifficulty($request->difficulty);
        }

        if ($request->has('voice')) {
            $query->byVoiceType($request->voice);
        }

        $tracks = $query->paginate(20);

        return response()->json([
            'tracks' => $tracks->items(),
            'pagination' => [
                'currentPage' => $tracks->currentPage(),
                'lastPage' => $tracks->lastPage(),
                'perPage' => $tracks->perPage(),
                'total' => $tracks->total(),
            ],
        ]);
    }

    public function show(LibraryTrack $track): JsonResponse
    {
        $track->load('category');

        return response()->json([
            'track' => [
                'id' => $track->id,
                'title' => $track->title,
                'slug' => $track->slug,
                'description' => $track->description,
                'transcript' => $track->transcript,
                'duration' => $track->duration,
                'formattedDuration' => $track->formatted_duration,
                'fileSize' => $track->file_size,
                'formattedFileSize' => $track->formatted_file_size,
                'voiceType' => $track->voice_type,
                'difficultyLevel' => $track->difficulty_level,
                'tags' => $track->tags,
                'safetyNotes' => $track->safety_notes,
                'playCount' => $track->play_count,
                'rating' => $track->rating,
                'category' => [
                    'id' => $track->category->id,
                    'name' => $track->category->name,
                    'slug' => $track->category->slug,
                    'color' => $track->category->color,
                ],
            ],
        ]);
    }

    public function search(Request $request): JsonResponse
    {
        $request->validate([
            'q' => 'required|string|min:2',
        ]);

        $query = $request->get('q');

        $tracks = LibraryTrack::active()
            ->with('category')
            ->where(function ($q) use ($query) {
                $q->where('title', 'ilike', "%{$query}%")
                  ->orWhere('description', 'ilike', "%{$query}%")
                  ->orWhere('transcript', 'ilike', "%{$query}%")
                  ->orWhereJsonContains('tags', [$query]);
            })
            ->orWhereHas('category', function ($q) use ($query) {
                $q->where('name', 'ilike', "%{$query}%");
            })
            ->orderBy('title')
            ->limit(20)
            ->get();

        return response()->json([
            'query' => $query,
            'tracks' => $tracks->map(function ($track) {
                return [
                    'id' => $track->id,
                    'title' => $track->title,
                    'slug' => $track->slug,
                    'description' => $track->description,
                    'duration' => $track->duration,
                    'formattedDuration' => $track->formatted_duration,
                    'category' => [
                        'name' => $track->category->name,
                        'slug' => $track->category->slug,
                        'color' => $track->category->color,
                    ],
                ];
            }),
        ]);
    }
}
