<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Services\LibraryService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class LibraryController extends Controller
{
    public function __construct(
        private LibraryService $libraryService
    ) {}

    public function categories(): JsonResponse
    {
        $categories = $this->libraryService->getCategories();
        
        return response()->json([
            'data' => $categories->map(function ($category) {
                return [
                    'id' => $category->id,
                    'name' => $category->name,
                    'slug' => $category->slug,
                    'description' => $category->description,
                    'icon' => $category->icon,
                    'color' => $category->color,
                    'tracksCount' => $category->tracks_count,
                ];
            }),
        ]);
    }

    public function tracks(Request $request): JsonResponse
    {
        $request->validate([
            'category_id' => 'nullable|integer|exists:library_categories,id',
            'voice_type' => 'nullable|string|in:male,female,neutral',
            'difficulty' => 'nullable|string|in:beginner,intermediate,advanced',
            'search' => 'nullable|string|max:255',
            'per_page' => 'nullable|integer|min:1|max:50',
        ]);

        $tracks = $this->libraryService->getTracks(
            $request->get('category_id'),
            $request->get('voice_type'),
            $request->get('difficulty'),
            $request->get('search'),
            $request->get('per_page', 12)
        );

        return response()->json([
            'data' => $tracks->items(),
            'pagination' => [
                'current_page' => $tracks->currentPage(),
                'last_page' => $tracks->lastPage(),
                'per_page' => $tracks->perPage(),
                'total' => $tracks->total(),
            ],
        ]);
    }

    public function track(string $slug): JsonResponse
    {
        $track = $this->libraryService->getTrackBySlug($slug);
        
        if (!$track) {
            return response()->json([
                'error' => 'Track not found',
            ], 404);
        }

        // Increment play count
        $this->libraryService->incrementPlayCount($track);

        return response()->json([
            'data' => [
                'id' => $track->id,
                'title' => $track->title,
                'slug' => $track->slug,
                'description' => $track->description,
                'transcript' => $track->transcript,
                'audioPath' => $track->audio_path,
                'imagePath' => $track->image_path,
                'durationSeconds' => $track->duration_seconds,
                'formattedDuration' => $track->formatted_duration,
                'tags' => $track->tags,
                'voiceType' => $track->voice_type,
                'difficultyLevel' => $track->difficulty_level,
                'playCount' => $track->play_count,
                'category' => [
                    'id' => $track->category->id,
                    'name' => $track->category->name,
                    'slug' => $track->category->slug,
                    'color' => $track->category->color,
                ],
                'createdAt' => $track->created_at,
                'updatedAt' => $track->updated_at,
            ],
        ]);
    }

    public function popular(): JsonResponse
    {
        $tracks = $this->libraryService->getPopularTracks(6);
        
        return response()->json([
            'data' => $tracks->map(function ($track) {
                return [
                    'id' => $track->id,
                    'title' => $track->title,
                    'slug' => $track->slug,
                    'description' => $track->description,
                    'durationSeconds' => $track->duration_seconds,
                    'formattedDuration' => $track->formatted_duration,
                    'playCount' => $track->play_count,
                    'category' => [
                        'name' => $track->category->name,
                        'color' => $track->category->color,
                    ],
                ];
            }),
        ]);
    }

    public function search(Request $request): JsonResponse
    {
        $request->validate([
            'query' => 'required|string|min:2|max:255',
            'per_page' => 'nullable|integer|min:1|max:50',
        ]);

        $tracks = $this->libraryService->searchTracks(
            $request->get('query'),
            $request->get('per_page', 12)
        );

        return response()->json([
            'data' => $tracks->items(),
            'pagination' => [
                'current_page' => $tracks->currentPage(),
                'last_page' => $tracks->lastPage(),
                'per_page' => $tracks->perPage(),
                'total' => $tracks->total(),
            ],
        ]);
    }
}
