<?php

namespace App\Services\Analysis;

use App\Models\Receipt;
use App\Services\AI\VertexClient;

class DiscountAnalysisService
{
    public function __construct(private VertexClient $vertex) {}

    /**
     * Analyze discounts on a receipt and link them to specific line items
     */
    public function analyzeDiscounts(Receipt $receipt): void
    {
        if (!$receipt->meta) {
            return;
        }

        $meta = is_string($receipt->meta) ? json_decode($receipt->meta, true) : $receipt->meta;
        
        if (!$meta) {
            return;
        }

        // Use LLM to analyze discounts and link them to line items
        $discountAnalysis = $this->analyzeDiscountsWithLLM($receipt, $meta);

        if ($discountAnalysis) {
            // Update the receipt meta with discount analysis
            $meta['discount_analysis'] = $discountAnalysis;
            $receipt->meta = $meta;
            $receipt->save();

            // Update individual line items with discount information
            $this->updateLineItemsWithDiscounts($receipt, $discountAnalysis);
        }
    }

    /**
     * Use LLM to intelligently analyze discounts and link them to line items
     */
    private function analyzeDiscountsWithLLM(Receipt $receipt, array $meta): ?array
    {
        $model = config('vertex.models.analyze', 'gemini-2.5-flash-lite');
        $system = 'You are an expert at analyzing receipt discounts and linking them to specific line items. Your task is to identify all discounts, promotions, and savings on a receipt and determine which line items they apply to.

DISCOUNT ANALYSIS RULES:
1. IDENTIFY ALL DISCOUNTS: Look for terms like "DISCOUNT", "SAVE", "OFF", "REDUCTION", "PROMOTION", "COUPON", "LOYALTY", "BULK", "MULTI-BUY"
2. DISCOUNT TYPES:
   - Percentage discounts (e.g., "10% OFF", "20% SAVINGS")
   - Fixed amount discounts (e.g., "£2 OFF", "SAVE £5")
   - Multi-buy discounts (e.g., "BUY 2 GET 1 FREE", "3 FOR 2")
   - Loyalty discounts (e.g., "LOYALTY 5% OFF")
   - Coupon discounts (e.g., "COUPON £3 OFF")
   - Bulk discounts (e.g., "BULK BUY DISCOUNT")
3. LINK TO LINE ITEMS: Determine which specific items the discount applies to
4. CALCULATE AMOUNTS: Work out the exact discount amount per item if split across multiple items
5. VAT IMPLICATIONS: Note if discounts affect VAT calculations

OUTPUT JSON STRUCTURE:
{
  "total_discount_amount": 5.50,
  "discount_breakdown": [
    {
      "discount_type": "percentage|fixed|multi_buy|loyalty|coupon|bulk",
      "discount_description": "10% OFF groceries",
      "discount_amount": 3.25,
      "applies_to_lines": [1, 2],
      "per_line_discount": {
        "1": 2.15,
        "2": 1.10
      },
      "vat_affected": true,
      "original_line_totals": {
        "1": 21.50,
        "2": 11.00
      },
      "discounted_line_totals": {
        "1": 19.35,
        "2": 9.90
      }
    }
  ],
  "discount_notes": "Applied 10% discount to grocery items, loyalty discount on fuel",
  "vat_recalculation_needed": true
}

IMPORTANT:
- If discount is split across multiple items, calculate proportional amounts
- If discount is item-specific, only apply to those items
- Always consider VAT implications of discounts
- Be precise with calculations to 2 decimal places
- If discount affects VAT, mark vat_recalculation_needed as true';

        $payload = [
            'receipt' => [
                'id' => $receipt->id,
                'merchant' => $receipt->merchant_name,
                'total_amount' => $receipt->total_amount,
                'vat_amount' => $meta['vat_amount'] ?? null,
                'subtotal' => $meta['subtotal'] ?? null,
                'discount_amount' => $meta['discount_amount'] ?? null,
                'lines' => $receipt->lines->map(fn($line) => [
                    'line_number' => $receipt->lines->search($line) + 1,
                    'description' => $line->description,
                    'quantity' => $line->quantity,
                    'unit_price' => $line->unit_price,
                    'line_total' => $line->line_total,
                    'vat_rate' => $line->vat_rate,
                    'meta' => is_string($line->meta) ? json_decode($line->meta, true) : $line->meta,
                ])->toArray(),
            ],
            'raw_receipt_data' => $meta,
        ];

        $response = $this->vertex->generate($model, json_encode($payload, JSON_UNESCAPED_SLASHES), $system, ['responseMimeType' => 'application/json']);
        
        if (!isset($response['json']) || !is_array($response['json'])) {
            return null;
        }

        return $response['json'];
    }

    /**
     * Update individual line items with discount information
     */
    private function updateLineItemsWithDiscounts(Receipt $receipt, array $discountAnalysis): void
    {
        if (empty($discountAnalysis['discount_breakdown'])) {
            return;
        }

        foreach ($discountAnalysis['discount_breakdown'] as $discount) {
            if (empty($discount['applies_to_lines'])) {
                continue;
            }

            foreach ($discount['applies_to_lines'] as $lineIndex) {
                // Convert 1-based line index to 0-based
                $lineNumber = $lineIndex - 1;
                $line = $receipt->lines->get($lineNumber);
                
                if (!$line) {
                    continue;
                }

                // Update line meta with discount information
                $lineMeta = is_string($line->meta) ? json_decode($line->meta, true) : $line->meta;
                if (!$lineMeta) {
                    $lineMeta = [];
                }

                $lineMeta['discount_info'] = [
                    'discount_type' => $discount['discount_type'],
                    'discount_description' => $discount['discount_description'],
                    'discount_amount' => $discount['per_line_discount'][$lineIndex] ?? 0,
                    'original_total' => $discount['original_line_totals'][$lineIndex] ?? $line->line_total,
                    'discounted_total' => $discount['discounted_line_totals'][$lineIndex] ?? $line->line_total,
                    'vat_affected' => $discount['vat_affected'] ?? false,
                ];

                $line->meta = $lineMeta;
                $line->save();
            }
        }
    }
}
