<div>
    @if($track)
    <div class="min-h-screen bg-gradient-to-br from-primary-50 to-accent-50">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
            <!-- Breadcrumb -->
            <nav class="flex mb-8" aria-label="Breadcrumb">
                <ol class="inline-flex items-center space-x-1 md:space-x-3">
                    <li class="inline-flex items-center">
                        <a href="{{ route('library.index') }}" class="inline-flex items-center text-sm font-medium text-gray-700 hover:text-primary-600">
                            <svg class="w-4 h-4 mr-2" fill="currentColor" viewBox="0 0 20 20">
                                <path d="M10.707 2.293a1 1 0 00-1.414 0l-7 7a1 1 0 001.414 1.414L4 10.414V17a1 1 0 001 1h2a1 1 0 001-1v-2a1 1 0 011-1h2a1 1 0 011 1v2a1 1 0 001 1h2a1 1 0 001-1v-6.586l.293.293a1 1 0 001.414-1.414l-7-7z"></path>
                            </svg>
                            Library
                        </a>
                    </li>
                    <li>
                        <div class="flex items-center">
                            <svg class="w-6 h-6 text-gray-400" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clip-rule="evenodd"></path>
                            </svg>
                            <a href="{{ route('library.category', $track->category->slug) }}" class="ml-1 text-sm font-medium text-gray-700 hover:text-primary-600 md:ml-2">
                                {{ $track->category->name }}
                            </a>
                        </div>
                    </li>
                    <li aria-current="page">
                        <div class="flex items-center">
                            <svg class="w-6 h-6 text-gray-400" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clip-rule="evenodd"></path>
                            </svg>
                            <span class="ml-1 text-sm font-medium text-gray-500 md:ml-2">{{ $track->title }}</span>
                        </div>
                    </li>
                </ol>
            </nav>

            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <!-- Main Content -->
                <div class="lg:col-span-2">
                    <!-- Track Header -->
                    <div class="bg-white rounded-2xl shadow-md p-6 mb-6">
                        <div class="flex items-start justify-between mb-4">
                            <div>
                                <h1 class="text-3xl font-bold text-gray-900 mb-2">{{ $track->title }}</h1>
                                <p class="text-lg text-gray-600 mb-4">{{ $track->description }}</p>
                                <div class="flex items-center space-x-4 text-sm text-gray-500">
                                    <span class="flex items-center">
                                        <svg class="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM9.555 7.168A1 1 0 008 8v4a1 1 0 001.555.832l3-2a1 1 0 000-1.664l-3-2z" clip-rule="evenodd"></path>
                                        </svg>
                                        {{ $track->duration }}
                                    </span>
                                    <span class="flex items-center">
                                        <svg class="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                            <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"></path>
                                        </svg>
                                        {{ $track->rating }}/5
                                    </span>
                                    <span class="flex items-center">
                                        <svg class="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20">
                                            <path fill-rule="evenodd" d="M3 4a1 1 0 011-1h12a1 1 0 110 2H4a1 1 0 01-1-1zm0 4a1 1 0 011-1h12a1 1 0 110 2H4a1 1 0 01-1-1zm0 4a1 1 0 011-1h12a1 1 0 110 2H4a1 1 0 01-1-1z" clip-rule="evenodd"></path>
                                        </svg>
                                        {{ $track->difficulty_level }}
                                    </span>
                                </div>
                            </div>
                            <div class="flex flex-col items-end">
                                <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-primary-100 text-primary-800 mb-2">
                                    {{ $track->category->name }}
                                </span>
                                <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-accent-100 text-accent-800">
                                    {{ $track->voice_type }}
                                </span>
                            </div>
                        </div>
                    </div>

                    <!-- Audio Player -->
                    <div class="bg-white rounded-2xl shadow-md p-6 mb-6">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Audio Player</h3>
                        @if($track->file_path && $track->streaming_url)
                            <div class="bg-gradient-to-r from-primary-100 to-accent-100 rounded-xl p-6">
                                <!-- Use the Audio Player Component -->
                                <x-audio-player :track="$track" :autoplay="false" />
                            </div>
                        @else
                            <div class="bg-gradient-to-r from-gray-100 to-gray-200 rounded-xl p-6 text-center">
                                <svg class="w-16 h-16 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19V6l12-3v13M9 19c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zm12-3c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zM9 10l12-3"></path>
                                </svg>
                                <p class="text-gray-600">No audio file available for this track.</p>
                                @if(!$track->file_path)
                                    <p class="text-sm text-gray-500 mt-2">File path not configured</p>
                                @elseif(!$track->streaming_url)
                                    <p class="text-sm text-gray-500 mt-2">Streaming URL could not be generated</p>
                                @endif
                            </div>
                        @endif
                    </div>

                    <!-- Transcript -->
                    @if($track->transcript)
                    <div class="bg-white rounded-2xl shadow-md p-6">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Transcript</h3>
                        <div class="prose max-w-none text-gray-700">
                            {!! nl2br(e($track->transcript)) !!}
                        </div>
                    </div>
                    @endif
                </div>

                <!-- Sidebar -->
                <div class="lg:col-span-1">
                    <!-- Related Tracks -->
                    <div class="bg-white rounded-2xl shadow-md p-6 mb-6">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Related Tracks</h3>
                        <div class="space-y-4">
                            @foreach($relatedTracks as $relatedTrack)
                            <a href="{{ route('library.track', $relatedTrack->slug) }}" class="block group">
                                <div class="flex items-center space-x-3 p-3 rounded-lg hover:bg-gray-50 transition-colors">
                                    <div class="flex-shrink-0 w-12 h-12 bg-gradient-to-br from-primary-100 to-accent-100 rounded-lg flex items-center justify-center">
                                        <svg class="w-6 h-6 text-primary-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19V6l12-3v13M9 19c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zm12-3c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zM9 10l12-3"></path>
                                        </svg>
                                    </div>
                                    <div class="flex-1 min-w-0">
                                        <p class="text-sm font-medium text-gray-900 group-hover:text-primary-600 truncate">
                                            {{ $relatedTrack->title }}
                                        </p>
                                        <p class="text-xs text-gray-500">{{ $relatedTrack->duration }}</p>
                                    </div>
                                </div>
                            </a>
                            @endforeach
                        </div>
                    </div>

                    <!-- Track Stats -->
                    <div class="bg-white rounded-2xl shadow-md p-6">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Track Statistics</h3>
                        <div class="space-y-3">
                            <div class="flex justify-between">
                                <span class="text-gray-600">Total Plays</span>
                                <span class="font-medium text-gray-900">{{ $track->play_count }}</span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-gray-600">Rating</span>
                                <span class="font-medium text-gray-900">{{ $track->rating }}/5</span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-gray-600">Difficulty</span>
                                <span class="font-medium text-gray-900">{{ Str::title($track->difficulty_level) }}</span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-gray-600">Voice Type</span>
                                <span class="font-medium text-gray-900">{{ Str::title($track->voice_type) }}</span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-gray-600">Added</span>
                                <span class="font-medium text-gray-900">{{ $track->created_at->format('M j, Y') }}</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Enhanced audio player functionality
        let audioPlayer;
        let isPlaying = false;
        let currentVolume = 100; // Initialize as percentage (0-100)
        let isMuted = false;
        let currentPlaybackRate = 1.0;
        let progressUpdateInterval;

        document.addEventListener('livewire:init', () => {
            audioPlayer = document.getElementById('audio-player');
            
            // Initialize audio player
            initializeAudioPlayer();
            
            // Listen for Livewire events
            Livewire.on('play-audio', (event) => {
                console.log('Playing audio for track:', event.trackId);
                playAudio();
            });

            Livewire.on('pause-audio', () => {
                console.log('Pausing audio');
                pauseAudio();
            });

            Livewire.on('stop-audio', () => {
                console.log('Stopping audio');
                stopAudio();
            });

            Livewire.on('seek-audio', (event) => {
                console.log('Seeking to:', event.time);
                seekTo(event.time);
            });

            Livewire.on('set-volume', (event) => {
                console.log('Setting volume to:', event.volume);
                setVolume(event.volume);
            });

            Livewire.on('set-playback-rate', (event) => {
                console.log('Setting playback rate to:', event.rate);
                setPlaybackRate(event.rate);
            });

            Livewire.on('audio-loaded', () => {
                console.log('Audio loaded successfully');
                Livewire.dispatch('audio-loaded');
            });

            Livewire.on('audio-error', (event) => {
                console.log('Audio error:', event.error);
                Livewire.dispatch('audio-error', { error: event.error });
            });

            Livewire.on('audio-ended', () => {
                console.log('Audio playback ended');
                Livewire.dispatch('audio-ended');
            });

            Livewire.on('audio-metadata-loaded', () => {
                console.log('Audio metadata loaded');
                updateDurationDisplay();
            });
        });

        function initializeAudioPlayer() {
            console.log('initializeAudioPlayer called');
            if (!audioPlayer) {
                console.error('Audio player element not found');
                return;
            }

            console.log('Audio player element found:', audioPlayer);

            // Set up event listeners
            audioPlayer.addEventListener('timeupdate', updateProgress);
            audioPlayer.addEventListener('loadedmetadata', updateDurationDisplay);
            audioPlayer.addEventListener('ended', onAudioEnded);
            audioPlayer.addEventListener('error', onAudioError);
            audioPlayer.addEventListener('loadstart', onAudioLoadStart);
            audioPlayer.addEventListener('canplay', onAudioCanPlay);

            // Set up progress slider
            const progressSlider = document.getElementById('progress-slider');
            if (progressSlider) {
                progressSlider.addEventListener('input', onProgressSliderChange);
                progressSlider.addEventListener('change', onProgressSliderChange);
            }

            // Set up volume slider
            const volumeSlider = document.getElementById('volume-slider');
            if (volumeSlider) {
                volumeSlider.addEventListener('input', onVolumeSliderChange);
            }

            // Set initial volume
            setVolume(100);
            
            // Update the volume slider to match the initial value
            if (volumeSlider) {
                volumeSlider.value = currentVolume;
            }
            
            // Update the volume text display
            if (volumeText) {
                volumeText.textContent = currentVolume + '%';
            }
            
            console.log('Audio player initialized successfully');
        }

        function togglePlayPause() {
            console.log('togglePlayPause called, isPlaying:', isPlaying);
            if (isPlaying) {
                pauseAudio();
            } else {
                playAudio();
            }
        }

        function playAudio() {
            console.log('playAudio called');
            if (!audioPlayer) {
                console.error('Audio player not found');
                return;
            }

            try {
                const playPromise = audioPlayer.play();
                
                if (playPromise !== undefined) {
                    playPromise.then(() => {
                        console.log('Audio started playing successfully');
                        isPlaying = true;
                        updatePlayPauseButton();
                        startProgressUpdates();
                        Livewire.dispatch('play-audio', { trackId: '{{ $track->id ?? "" }}' });
                    }).catch(error => {
                        console.error('Error playing audio:', error);
                        onAudioError(error);
                    });
                }
            } catch (error) {
                console.error('Error playing audio:', error);
                onAudioError(error);
            }
        }

        function pauseAudio() {
            console.log('pauseAudio called');
            if (!audioPlayer) {
                console.error('Audio player not found');
                return;
            }

            audioPlayer.pause();
            isPlaying = false;
            updatePlayPauseButton();
            stopProgressUpdates();
            console.log('Audio paused successfully');
            Livewire.dispatch('pause-audio');
        }

        function stopAudio() {
            if (!audioPlayer) return;

            audioPlayer.pause();
            audioPlayer.currentTime = 0;
            isPlaying = false;
            updatePlayPauseButton();
            stopProgressUpdates();
            updateProgressDisplay();
            Livewire.dispatch('stop-audio');
        }

        function skipTime(seconds) {
            if (!audioPlayer) return;

            const newTime = Math.max(0, Math.min(audioPlayer.duration, audioPlayer.currentTime + seconds));
            audioPlayer.currentTime = newTime;
            updateProgressDisplay();
        }

        function seekTo(time) {
            if (!audioPlayer) return;

            audioPlayer.currentTime = time;
            updateProgressDisplay();
        }

        function setVolume(volume) {
            console.log('setVolume called with:', volume);
            if (!audioPlayer) {
                console.error('Audio player not found');
                return;
            }

            // Store the volume as a percentage (0-100)
            currentVolume = Math.max(0, Math.min(100, volume));
            // Set the audio volume as a decimal (0-1)
            audioPlayer.volume = currentVolume / 100;
            
            console.log('Volume set to:', currentVolume + '% (audio volume:', audioPlayer.volume + ')');
            
            // Update UI
            const volumeSlider = document.getElementById('volume-slider');
            const volumeText = document.getElementById('volume-text');
            
            if (volumeSlider) volumeSlider.value = currentVolume;
            if (volumeText) volumeText.textContent = currentVolume + '%';
            
            // Update mute state
            isMuted = currentVolume === 0;
            updateMuteButton();
            
            Livewire.dispatch('set-volume', { volume: currentVolume / 100 });
        }

        function toggleMute() {
            console.log('toggleMute called, isMuted:', isMuted, 'currentVolume:', currentVolume);
            if (isMuted) {
                // Unmute by restoring the previous volume
                setVolume(currentVolume);
            } else {
                // Mute by setting volume to 0
                setVolume(0);
            }
            Livewire.dispatch('toggle-mute');
        }

        function setPlaybackRate(rate) {
            if (!audioPlayer) return;

            currentPlaybackRate = parseFloat(rate);
            audioPlayer.playbackRate = currentPlaybackRate;
            Livewire.dispatch('set-playback-rate', { rate: currentPlaybackRate });
        }

        function updatePlayPauseButton() {
            const playIcon = document.getElementById('play-icon');
            const pauseIcon = document.getElementById('pause-icon');
            
            if (isPlaying) {
                if (playIcon) playIcon.classList.add('hidden');
                if (pauseIcon) pauseIcon.classList.remove('hidden');
            } else {
                if (playIcon) playIcon.classList.remove('hidden');
                if (pauseIcon) pauseIcon.classList.add('hidden');
            }
        }

        function updateMuteButton() {
            const volumeIcon = document.getElementById('volume-icon');
            const muteIcon = document.getElementById('mute-icon');
            
            if (isMuted) {
                if (volumeIcon) volumeIcon.classList.add('hidden');
                if (muteIcon) muteIcon.classList.remove('hidden');
            } else {
                if (volumeIcon) volumeIcon.classList.remove('hidden');
                if (muteIcon) muteIcon.classList.add('hidden');
            }
        }

        function updateProgress() {
            if (!audioPlayer) return;

            const currentTime = audioPlayer.currentTime;
            const duration = audioPlayer.duration;
            
            if (duration > 0) {
                const progressPercent = (currentTime / duration) * 100;
                updateProgressBar(progressPercent);
                updateTimeDisplay(currentTime, duration);
                
                // Update Livewire component
                Livewire.dispatch('update-progress', { 
                    currentTime: currentTime, 
                    duration: duration 
                });
            }
        }

        function updateProgressBar(percent) {
            const progressBar = document.getElementById('progress-bar');
            if (progressBar) {
                progressBar.style.width = percent + '%';
            }
        }

        function updateTimeDisplay(currentTime, duration) {
            const currentTimeDisplay = document.getElementById('current-time');
            const totalDurationDisplay = document.getElementById('total-duration');
            
            if (currentTimeDisplay) {
                currentTimeDisplay.textContent = formatTime(currentTime);
            }
            
            if (totalDurationDisplay && duration) {
                totalDurationDisplay.textContent = formatTime(duration);
            }
        }

        function formatTime(seconds) {
            if (isNaN(seconds)) return '0:00';
            
            const minutes = Math.floor(seconds / 60);
            const remainingSeconds = Math.floor(seconds % 60);
            return `${minutes}:${remainingSeconds.toString().padStart(2, '0')}`;
        }

        function updateDurationDisplay() {
            if (!audioPlayer) return;

            const duration = audioPlayer.duration;
            if (duration && !isNaN(duration)) {
                updateTimeDisplay(audioPlayer.currentTime, duration);
            }
        }

        function startProgressUpdates() {
            if (progressUpdateInterval) {
                clearInterval(progressUpdateInterval);
            }
            progressUpdateInterval = setInterval(updateProgress, 100);
        }

        function stopProgressUpdates() {
            if (progressUpdateInterval) {
                clearInterval(progressUpdateInterval);
                progressUpdateInterval = null;
            }
        }

        function onProgressSliderChange(event) {
            if (!audioPlayer) return;

            const percent = event.target.value;
            const newTime = (percent / 100) * audioPlayer.duration;
            audioPlayer.currentTime = newTime;
            updateProgressDisplay();
        }

        function onVolumeSliderChange(event) {
            console.log('Volume slider changed to:', event.target.value);
            setVolume(parseInt(event.target.value));
        }

        function updateProgressDisplay() {
            updateProgress();
        }

        function onAudioEnded() {
            isPlaying = false;
            updatePlayPauseButton();
            stopProgressUpdates();
            Livewire.dispatch('audio-ended');
        }

        function onAudioError(error) {
            console.error('Audio error:', error);
            isPlaying = false;
            updatePlayPauseButton();
            stopProgressUpdates();
            Livewire.dispatch('audio-error', { error: error.message || 'Audio playback error' });
        }

        function onAudioLoadStart() {
            console.log('Audio loading started');
        }

        function onAudioCanPlay() {
            console.log('Audio can start playing');
            updateDurationDisplay();
        }

        // Clean up on page unload
        window.addEventListener('beforeunload', () => {
            if (progressUpdateInterval) {
                clearInterval(progressUpdateInterval);
            }
        });
    </script>

    <style>
    .slider {
        -webkit-appearance: none;
        appearance: none;
        background: transparent;
        cursor: pointer;
    }

    .slider::-webkit-slider-track {
        background: #d1d5db;
        height: 8px;
        border-radius: 4px;
    }

    .slider::-webkit-slider-thumb {
        -webkit-appearance: none;
        appearance: none;
        background: #4dbda0;
        height: 16px;
        width: 16px;
        border-radius: 50%;
        cursor: pointer;
        border: 2px solid #ffffff;
        box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        transition: all 0.2s ease;
    }

    .slider::-webkit-slider-thumb:hover {
        background: #3e9780;
        transform: scale(1.1);
    }

    .slider::-moz-range-track {
        background: #d1d5db;
        height: 8px;
        border-radius: 4px;
        border: none;
    }

    .slider::-moz-range-thumb {
        background: #4dbda0;
        height: 16px;
        width: 16px;
        border-radius: 50%;
        cursor: pointer;
        border: 2px solid #ffffff;
        box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        transition: all 0.2s ease;
    }

    .slider::-moz-range-thumb:hover {
        background: #3e9780;
        transform: scale(1.1);
    }
    </style>
    @endif
    @endif
</div>
